import logging
import os
import time

from datetime import datetime


class Cleaner(object):
    def __init__(self, yt_cluster, yt_token=None, dry_run=False, batch_execute=True):
        self.__init_yt(yt_cluster, yt_token)
        self._dry_run = dry_run
        self._batch_execute = batch_execute
        self.__now = time.time()
        logging.info('Now: {}'.format(self.__now))

    def __init_yt(self, cluster, token):
        from yt.wrapper import YtClient
        self.__yt_cluster = cluster
        self.__yt = YtClient(proxy=cluster, token=token)
        self.__yt_batch = self.__yt.create_batch_client(raise_errors=True)
        logging.getLogger('Yt').setLevel(logging.DEBUG)

    def _error(self, msg):
        raise Exception('{}: {}'.format(self.__yt_cluster, msg))

    def _remove_node(self, path, expl):
        logging.info('Remove path={} expl: {}'.format(path, expl))
        if not self._dry_run:
            if self._batch_execute:
                self.__yt_batch.remove(path)
            else:
                self.__yt.remove(path)

    def _run(self, path, ttl):
        items = self.__yt.list(path, attributes=['type'])
        childs = len(items)
        for item in items:
            item_path = os.path.join(path, item)
            if item[0] == '.':
                logging.info('Skip system node: {}'.format(item_path))
                continue

            if item.attributes['type'] == 'table':
                if item.endswith('-300') and len(item) == 14:
                    try:
                        ts = int(item.split('-')[0])
                    except Exception as e:
                        self._error('Cannot get timestamp from table name path={} error={}'.format(item_path, e))
                    age = self.__now - ts
                    if age > ttl:
                        try:
                            self._remove_node(
                                item_path,
                                'by ttl age={:.3f}d time={}'.format(
                                    age / (24.0 * 60 * 60),
                                    datetime.fromtimestamp(ts).strftime('%Y-%m-%dT%H:%M:%S')
                                )
                            )
                            childs -= 1
                        except Exception as e:
                            logging.error('Removing of %s failed due to %s', item_path, e)
                else:
                    self._error('Unknown table name format path={}'.format(item_path))
            elif item.attributes['type'] == 'map_node':
                item_childs = self._run(item_path, ttl)
                if item_childs == 0:
                    self._remove_node(item_path, 'empty map_node')
                    childs -= 1
            else:
                self._error('Unsupported node type={} path={}'.format(
                    item.attributes['type'],
                    item_path
                ))
        if self._batch_execute:
            self.__yt_batch.commit_batch()
        return childs

    def run(self, lb_cluster, ttl):
        self._run('//home/saas-lb-delivery/{}'.format(lb_cluster), ttl * 24 * 60 * 60)


def logs_to_stdout():
    import sys
    logging.basicConfig(stream=sys.stdout, level=logging.DEBUG)


if __name__ == '__main__':
    logs_to_stdout()

    import argparse
    parser = argparse.ArgumentParser()
    parser.add_argument('-l', '--lb-cluster', default='logbroker')
    parser.add_argument('-y', '--yt-cluster', default='arnold')
    parser.add_argument('--ttl', type=int, default=14)
    parser.add_argument('--apply', action='store_true', default=False)
    args = parser.parse_args()

    cleaner = Cleaner(
        args.yt_cluster,
        dry_run=not args.apply
    )
    cleaner.run(args.lb_cluster, args.ttl)
