# coding: utf-8

import traceback
import json
import datetime
import os

import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as parameters
import sandbox.common.errors as errors
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.projects.gencfg.helpers as helpers
import sandbox.projects.gencfg.environment as environment

from sandbox.projects.saas.common.resources import SaasHostsInfoFull


class CollectSaasCloudHostsInfo(sdk2.Task):
    """Collect SaaS hosts by requirements"""

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD

    class Context(sdk2.Task.Context):
        exceptions = []
        gencfg_revision = -1
        retry_count = 0

    class Parameters(sdk2.Task.Parameters):
        with parameters.Group('Execution') as execution:
            arcadia_revision = sdk2.parameters.String(
                'Base revision',
                description='Allocate, check and commit against this revision',
                required=False
            )
            use_last_resources = sdk2.parameters.Bool(
                'Use last released gencfg resources',
                required=True,
                default=False
            )

    def run_prepare(self, gencfg, params):
        gencfg.prepare()
        gencfg.install(params.use_last_resources)

    def on_execute(self):
        # Good idea to import from lib23/gencfg.py, but a conflict of protobuf occurs
        GENCFG_GROUPS_LOCATIONS = frozenset({'SAS', 'MAN', 'VLA', 'IVA'})

        self.Context.retry_count += 1
        date_str = datetime.date.today().isoformat()
        result_file_name = '{}.json'.format(date_str)
        try:
            gencfg = environment.GencfgEnvironment(self, self.Parameters.arcadia_revision, self.ramdrive.path)
            self.run_prepare(gencfg, self.Parameters)
            self.Context.gencfg_revision = int(gencfg.info(gencfg.src_root)['commit_revision'])
            self.set_info('Task manipulates revision {}'.format(self.Context.gencfg_revision))
        except Exception as e:
            helpers.print_log('\n' + traceback.format_exc(limit=20))
            raise Exception('Task failed at prepare gencfg environment.\n[{}]: {}'.format(type(e).__name__, e))
        try:
            raw_hosts_data = []
            for geo in GENCFG_GROUPS_LOCATIONS:
                group_name = '{}_SAAS_CLOUD'.format(geo.upper())
                data_file_name = os.path.join(os.getcwd(), '{}.json'.format(group_name))
                self.dump_group_data(gencfg, group_name, data_file_name)
                with open(str(data_file_name)) as f:
                    raw_hosts_data.extend(json.load(f))

            resource = sdk2.ResourceData(SaasHostsInfoFull(self, "SaasHostsInfoFull", result_file_name))
            with open(str(resource.path), 'w') as f:
                json.dump(raw_hosts_data, f)
            resource.ready()
        except Exception as e:
            helpers.print_log('\n' + traceback.format_exc(limit=20))
            raise errors.TaskFailure('[{}]: {}'.format(type(e).__name__, e))

    @staticmethod
    def list_to_comma_separated(lst):
        return ','.join(map(str, lst))

    @staticmethod
    def dump_group_data(gencfg, group, filename):
        collect_hosts_command = [
            './utils/saas/show_hosts_info.py',
            '--yt',
            '--group', group,
            '--file', str(filename)
            ]

        gencfg.run_process(collect_hosts_command, 'collect_hosts_command')
