# coding: utf-8

import six
import json
import datetime

import sandbox.sdk2 as sdk2
import sandbox.common.types.task as ctt

from sandbox.projects.saas.common.resources import SaasHostsInfoFull
from sandbox.projects.saas.CollectSaasCloudHostsInfo import CollectSaasCloudHostsInfo


class CollectSaasCloudStats(sdk2.Task):
    """Collect SaaS cloud statistics"""

    class Parameters(sdk2.Task.Parameters):
        force_recollect = sdk2.parameters.Bool(
            'Force recollect',
            description='Don\'t use existing SaasHostsInfoFull if any',
            required=True, default=False
        )

    def on_execute(self):
        # import yt staff
        import yt.wrapper as yt
        import statface_client
        from saas.tools.devops.lib_sandbox.saas_cloud_gencfg_stats import YT_CLUSTER, SLOT_FLAVORS, INSTANCES_TABLE_LINK, HOSTS_TABLE_LINK
        from saas.tools.devops.lib_sandbox.saas_cloud_gencfg_stats import get_hosts_table, get_instance_table, prepare_yt_data
        yt.config.set_proxy(YT_CLUSTER)
        yt.config.config['token'] = sdk2.Vault.data('SAAS-ROBOT', 'YT_TOKEN_ARNOLD')

        statface = statface_client.StatfaceClient(
            host=statface_client.STATFACE_PRODUCTION,
            oauth_token=sdk2.Vault.data('SAAS-ROBOT', 'STAT_SAAS_DEVOPS_TOKEN')
        )

        resource = SaasHostsInfoFull.find(
            attrs=dict(date=datetime.date.today().isoformat()),
        ).first()
        if self.Parameters.force_recollect or resource is None:
            with self.memoize_stage.create_children:
                sub_task = CollectSaasCloudHostsInfo(
                    self,
                    description="Collect gencfg info for YT analyse in task {}".format(self.id),
                    use_last_resources=True,
                )
                sub_task.Requirements.tasks_resource = self.Requirements.tasks_resource
                sub_task.save()
                sub_task.enqueue()
                raise sdk2.WaitTask([sub_task, ], ctt.Status.Group.FINISH, wait_all=True, timeout=3600)

            sub_task = self.find(CollectSaasCloudHostsInfo, status=ctt.Status.Group.FINISH).first()
            resource = SaasHostsInfoFull.find(
                task=sub_task,
            ).first()
        # get resource
        info_date = datetime.date(*map(int, resource.date.split('-')))
        resource_data = sdk2.ResourceData(resource)
        resource_path = resource_data.path
        with open(str(resource_path)) as f:
            host_data = json.load(f)

        host_data, instance_data = prepare_yt_data(host_data)

        host_table = get_hosts_table(info_date)
        yt.write_table(host_table, host_data, force_create=True)
        yt.link(target_path=host_table, link_path=HOSTS_TABLE_LINK, force=True)

        instance_table = get_instance_table(info_date)
        yt.write_table(instance_table, instance_data, force_create=True)
        yt.link(target_path=instance_table, link_path=INSTANCES_TABLE_LINK, force=True)

        # TODO: move to saas
        stats_data = {}
        for h in host_data:
            dc = h['dc']
            if stats_data.get(dc) is None:
                stats_data[dc] = {}
                for indexing in ('true', 'false'):
                    stats_data[dc][indexing] = {}
                    for slot_flavor in SLOT_FLAVORS:
                        stats_data[dc][indexing][slot_flavor] = 0
            for k, v in six.iteritems(SLOT_FLAVORS):
                if h['ram_available'] >= v['mem'] and h['cpu_power_available'] >= v['cpu']:
                    if h['noindexing'] is None:
                        stats_data[dc]['true'][k] += 1
                        stats_data[dc]['false'][k] += 1
                    else:
                        stats_data[dc][str(h['noindexing']).lower()][k] += 1

        stats_report = []
        for dc, values in six.iteritems(stats_data):
            for no_indexing, values_1 in six.iteritems(values):
                base_dict = {'fielddate': resource.date, 'dc': dc, 'no_indexing': no_indexing}
                for slot_flavor, slot_number in six.iteritems(values_1):
                    base_dict[slot_flavor] = slot_number
                stats_report.append(base_dict)
        report = statface.get_report('SAAS/resources/Slots')
        report.upload_data(scale='daily', data=stats_report)
