# coding: utf-8

import traceback

import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as parameters
import sandbox.common.errors as errors

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.projects.gencfg.helpers as helpers
import sandbox.projects.gencfg.environment as environment

import sandbox.projects.saas.common.semaphores as semaphores
from sandbox.projects.saas.common.resources import SaaSHostsInfo


class CollectSaasHosts(sdk2.Task):
    """Collect SaaS hosts by requirements"""

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD
        semaphores = semaphores.SaasSemaphore.gencfg

    class Context(sdk2.Task.Context):
        exceptions = []
        gencfg_revision = -1
        retry_count = 0

    class Parameters(sdk2.Task.Parameters):
        with parameters.Group('Request params') as request_params:
            group_name = parameters.String('GenCfg group', required=True)
            cpu_requirements = parameters.Integer('Required CPU power in Kimkim power units', required=True)
            mem_requirements = parameters.Integer('Required RAM in Gb', required=True)
            ssd_requirements = parameters.Bool('Only hosts with SSD', default=False, required=True)
            no_indexing = parameters.Bool('Only NO_INDEXING hosts', default=False, required=True)
            exclude_groups = parameters.List('Exclude hosts present in Groups')

        with parameters.Group('Execution') as execution:
            arcadia_revision = sdk2.parameters.String(
                'Base revision',
                description='Allocate, check and commit against this revision',
                required=False
            )
            use_last_resources = sdk2.parameters.Bool(
                'Use last released resources',
                required=True,
                default=False
            )

    def run_prepare(self, gencfg, params):
        gencfg.prepare()
        gencfg.install(params.use_last_resources)

    def on_execute(self):
        self.Context.retry_count += 1

        try:
            gencfg = environment.GencfgEnvironment(self, self.Parameters.arcadia_revision, self.ramdrive.path)
            self.run_prepare(gencfg, self.Parameters)
            self.Context.gencfg_revision = int(gencfg.info(gencfg.src_root)['commit_revision'])
            self.set_info('Task manipulates revision {}'.format(self.Context.gencfg_revision))
        except Exception as e:
            helpers.print_log('\n' + traceback.format_exc(limit=20))
            raise Exception('Task failed at prepare gencfg environment.\n[{}]: {}'.format(type(e).__name__, e))

        try:
            resource = sdk2.ResourceData(SaaSHostsInfo(self, "SaaSHostsInfo", "result.json"))
            self.collect_hosts(gencfg, self.Parameters, resource_path=resource.path)
            resource.ready()
            return
        except Exception as e:
            helpers.print_log('\n' + traceback.format_exc(limit=20))
            raise errors.TaskFailure('[{}]: {}'.format(type(e).__name__, e))

    @staticmethod
    def list_to_comma_separated(lst):
        return ','.join(map(str, lst))

    def collect_hosts(self, gencfg, params, resource_path):
        collect_hosts_command = [
            './utils/saas/collect_saas_hosts.py',
            '--group', params.group_name,
            '--slot-size-cpu', str(params.cpu_requirements),
            '--slot-size-mem', str(params.mem_requirements),
            '--file', str(resource_path)
            ]
        if params.ssd_requirements:
            collect_hosts_command.append('--ssd')
        if params.no_indexing:
            collect_hosts_command.append('--no-indexing')
        if params.exclude_groups:
            collect_hosts_command.extend(['--exclude-groups', self.list_to_comma_separated(params.exclude_groups)])

        gencfg.run_process(collect_hosts_command, 'collect_hosts_command')
