# coding: utf-8

import sandbox.sdk2 as sdk2
from sandbox.projects.saas.common.classes import SaasBinaryTask
import sandbox.sandboxsdk.svn as sdk_svn
import logging
import yaml
from collections import OrderedDict
from sandbox.sandboxsdk import ssh

ARCADIA_ITS_PATH = "/infra/awacs/templates/aux-balancers-data/"


class CommitToArcadiaIts(SaasBinaryTask):

    TASKS_RESOURCE_NAME = 'CommitToArcadiaItsTask'

    class Parameters(sdk2.Task.Parameters):
        namespace_id = sdk2.parameters.String('Id of target namespace', required=True)
        by_balancer = sdk2.parameters.List('Tags for ITS by balancer', required=False, default=["service_balancer_off"])
        common = sdk2.parameters.List('Tags for common ITS', required=False, default=["cplb_balancer_load_switch"])
        dont_off_balancers_by_marty = sdk2.parameters.Bool("Don't off balancers by Marty", required=False)
        balancer_its_dir = sdk2.parameters.ArcadiaUrl(
            'Path to balancers ITS configs directory',
            required=True,
            default_value=sdk_svn.Arcadia.trunk_url() + ARCADIA_ITS_PATH
        )

    def svn_add(self, file_path):
        try:
            sdk2.svn.Arcadia.add(file_path)
        except sdk_svn.SvnError:
            logging.error('Path %s already added to svn' % file_path)

    def ordered_load(self, stream, Loader=yaml.Loader, object_pairs_hook=OrderedDict):
        class OrderedLoader(Loader):
            pass

        def construct_mapping(loader, node):
            loader.flatten_mapping(node)
            return object_pairs_hook(loader.construct_pairs(node))

        OrderedLoader.add_constructor(
            yaml.resolver.BaseResolver.DEFAULT_MAPPING_TAG,
            construct_mapping)
        return yaml.load(stream, OrderedLoader)

    def ordered_dump(self, data, stream=None, Dumper=yaml.Dumper, **kwds):
        class OrderedDumper(Dumper):
            pass

        def _dict_representer(dumper, data):
            return dumper.represent_mapping(
                yaml.resolver.BaseResolver.DEFAULT_MAPPING_TAG,
                data.items())

        OrderedDumper.add_representer(OrderedDict, _dict_representer)
        return yaml.dump(data, stream, OrderedDumper, **kwds)

    def commit(self, message):
        with ssh.Key(self, 'RTYSERVER-ROBOT', 'ssh_key'):
            return sdk2.svn.Arcadia.commit(self.Context.local_path, message, user='saas-robot')

    def on_prepare(self):
        sdk2.svn.Arcadia.checkout(self.Parameters.balancer_its_dir, 'data')

    def on_execute(self):
        self.Context.local_path = "data/data.yaml"
        with open(self.Context.local_path, "r") as stream:
            data = self.ordered_load(stream)
        assert data
        data[str(self.Parameters.namespace_id)] = {'by_balancer': [str(el) for el in self.Parameters.by_balancer],
                                                   'common': [str(el) for el in self.Parameters.common]}
        if self.Parameters.dont_off_balancers_by_marty:
            data[str(self.Parameters.namespace_id)]['dont_off_balancers_by_marty'] = True
        with open(self.Context.local_path, "w") as stream:
            self.ordered_dump(data, stream)
        self.svn_add(self.Context.local_path)
        self.Context.svn_diff = sdk2.svn.Arcadia.diff(self.Context.local_path)
        self.commit('SKIP_CHECK  Adding ITS config for namespace {}'.format(self.Parameters.namespace_id))
