# -*- coding: utf-8 -*-

import logging
import requests
import re
import time
import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.projects.saas.common.classes import SaasBinaryTask


class DeployFerrymanConfig(SaasBinaryTask, nanny.ReleaseToNannyTask2):
    """
    Deploy configs SaaS Ferryman by revision
    Required commit with ferryman config changes
    """
    TASKS_RESOURCE_NAME = 'SaasSandboxTasks'

    class Parameters(sdk2.Parameters):
        revison_num = sdk2.parameters.Integer('Revison number', required=True)

    @staticmethod
    def get_revision_id_from_message(message):
        r = re.compile(r' \d+')
        result = r.search(message)
        if result:
            result = result.group().strip()
        else:
            result = ''
        return result

    @staticmethod
    def get_sandbox_task_instance(sandbox_task_id):
        logging.info('Trying to get sandbox instance with id %s', sandbox_task_id)
        subtask = sdk2.Task.find(id=sandbox_task_id).first()
        while subtask.status not in [ctt.Status.SUCCESS, ctt.Status.RELEASED]:
            logging.info('Waiting for task %s, current status %s', sandbox_task_id, subtask.status)
            time.sleep(60)
            subtask = sdk2.Task.find(id=sandbox_task_id).first()
        return subtask

    @staticmethod
    def nanny_activate_release(sandbox_task_id):
        from nanny_rpc_client.exceptions import BadRequestError
        from saas.tools.devops.lib23.nanny_helpers import NannyReleaseRunner
        from nanny_tickets import tickets_api_pb2

        nanny_token = sdk2.Vault.data('RTYSERVER-ROBOT', 'nanny_token')
        nanny_releaser = ''
        while not nanny_releaser:
            try:
                nanny_releaser = NannyReleaseRunner(sandbox_task_id=str(sandbox_task_id),
                                                    nanny_oauth_token=nanny_token)
            except AssertionError:
                logging.warning('Nanny release is not ready yet')
                time.sleep(60)
        while not nanny_releaser.tickets:
            logging.info('Waiting for release tickets')
            time.sleep(60)
            nanny_releaser = NannyReleaseRunner(sandbox_task_id=str(sandbox_task_id),
                                                nanny_oauth_token=nanny_token)
        for ticket in nanny_releaser.tickets:
            try:
                logging.info('Activating ticket %s', ticket)
                request = tickets_api_pb2.CreateTicketEventRequest()
                request.ticket_id = ticket.id
                request.spec.type = 5  # ACTIVATE_SERVICE_SNAPSHOT
                request.spec.activate_snapshot.activate_recipe = 'default'  # default reploy reсipe for Ferrymans
                response = nanny_releaser.tickets_stub.create_ticket_event(request)
                logging.debug(response)
            except BadRequestError as e:
                logging.exception(e.message)

    def get_sandbox_task_id(self, testenv_answer, revision=None):
        sandbox_id = ''
        for row in testenv_answer['rows']:
            if (revision and int(revision) == row['revision']) or self.Parameters.startrek_ticket in row['revision_info/comment']:
                if row['task_id']:
                    sandbox_id = row['task_id']
                    break
        if sandbox_id:
            logging.info('Target sandbox id %s', sandbox_id)
        return sandbox_id

    def get_id_from_testenv(self, commit_revision=''):
        db_name = 'rtyserver-trunk-saas-ferryman'
        test_name = 'BUILD_FERRYMAN_CONFIGS'
        request_url = 'https://testenv.yandex-team.ru/handlers/grids/testResults?' \
                      'database={}&test_name={}&hide_filtered=true&revision={}'.format(db_name, test_name, commit_revision)
        sandbox_task_id = ''
        while not sandbox_task_id:
            response = requests.get(request_url, verify=False).json()
            logging.debug(response)
            sandbox_task_id = self.get_sandbox_task_id(response, revision=commit_revision)
            time.sleep(60)
        return sandbox_task_id

    def on_execute(self):
        rev_id = str(self.Parameters.revison_num)
        logging.info('Revision: ' + rev_id)
        # Get sandbox task
        subtask_id = self.get_id_from_testenv(rev_id)
        # Wait for task before release
        self.get_sandbox_task_instance(subtask_id)
        # Release task
        logging.info('Create release')
        self.server.release(task_id=subtask_id,
                            type=ctt.ReleaseStatus.STABLE,
                            subject='Released task in STABLE')
        time.sleep(30)
        self.server.release(task_id=subtask_id,
                            type=ctt.ReleaseStatus.PRESTABLE,
                            subject='Released task in PRESTABLE')
        # Deploy
        self.nanny_activate_release(subtask_id)
