# coding: utf-8

import math
import logging
import sandbox.sdk2 as sdk2

from sandbox.projects.saas.common.classes import SaasBinaryTask
from sandbox.projects.saas.common.resources import SaasTankAmmo

class MakeAmmoFromYtLogs(SaasBinaryTask):
    """Make Tank ammo from saas logs in YT"""
    TASKS_RESOURCE_NAME = 'SaasLoadTestTasks'

    class Parameters(sdk2.Task.Parameters):
        ctype = sdk2.parameters.String('SaaS ctype', required=True, hint=True)
        service = sdk2.parameters.String('SaaS service name', required=True, hint=True)
        replace_service = sdk2.parameters.String('Replace service name', required=False, default_value='')
        strip_reqid = sdk2.parameters.Bool('Strip reqid from requests', default_value=True)
        magazine_size = sdk2.parameters.Integer('Desired ammo count', default_value=5000)
        yt_proxy = sdk2.parameters.String('YT cluster', default_value='hahn')
        tmp_dir_path = sdk2.parameters.String('YT tmp dir path', default_value='//home/saas/tmp')

        with sdk2.parameters.Group('Authorisation') as authorisation:
            yt_secret_id = sdk2.parameters.String('YAV secret id with YT token', default_value='sec-01d74p6hkcabgfq4cv38mqyn9k')
            yt_secret_key = sdk2.parameters.String('YAV secret key for YT', default_value='yt_token')
            tvm_secret_id = sdk2.parameters.String('YAV secret id with TVM token', default_value='sec-01ehywawrh8mepfexm5mq2dg55')
            tvm_secret_key = sdk2.parameters.String('YAV secret key for TVM', default_value='client_secret')

    def prepare_yt(self):
        import yt.wrapper as yt
        yt_token_secret = sdk2.yav.Secret(self.Parameters.yt_secret_id)
        yt.config.set_proxy(self.Parameters.yt_proxy)
        yt.config.config['token'] = yt_token_secret.data()[self.Parameters.yt_secret_key]
        return yt

    def start_log_processing_operation(self, source_table, destination_table, sync=False):
        from saas.library.python.yt_tools.log_to_ammo_mapper import SaasLog2Ammo
        yt = self.prepare_yt()
        return yt.run_map(
            SaasLog2Ammo(
                self.Parameters.service,
                replace_service=self.Parameters.replace_service if self.Parameters.replace_service else None,
                strip_reqid=self.Parameters.strip_reqid,
                reqinfo='saas-ammo-task-{}'.format(self.id)
            ),
            source_table=source_table,
            destination_table=destination_table,
            spec={
                'auto_merge': {'mode': 'relaxed'},
                'data_size_per_job': 512 * 2 ** 20  # 512 MB => ~ 2 minutes per job average
            },
            sync=sync
        )

    def on_execute(self):
        from saas.library.python.yt_tools.common import attribute_path, canonical_path
        import logging

        yt = self.prepare_yt()
        output_table_path = "{}/{}".format(self.Parameters.tmp_dir_path, self.id)

        self.Context.yt_path = '//logs/saas/{}/access-log/30min'.format(self.Parameters.ctype)

        if not self.Context.yt_operation:
            self.Context.node_type = yt.get(attribute_path(self.Context.yt_path, 'type'))
            if self.Context.node_type == 'table':
                yt_path = [canonical_path(self.Context.yt_path), ]
            elif self.Context.node_type == 'map_node':
                yt_path = yt.list(canonical_path(self.Context.yt_path), absolute=True)
                yt_path = list(reversed(yt_path))
            else:
                raise RuntimeError('Unsupported YT path {}'.format(self.Context.yt_path))

            yt_operation = self.start_log_processing_operation(yt_path[0], output_table_path)
            self.Context.yt_operation = yt_operation.id
            logging.info('YT operation %s started', yt_operation.url)

        yt_operation = yt.Operation(self.Context.yt_operation)
        logging.info('YT operation progress: %s', yt_operation.get_progress())

        while yt_operation.get_state() not in ('completed', 'failed'):
            logging.info('Wait YT operation finish iteration')
            raise sdk2.WaitTime(60)  # minute

        if yt_operation.get_state() != 'completed':
            raise RuntimeError('YT operation {}, details: {}'.format(yt_operation.get_state(), yt_operation.url))

        result_row_count = yt.get(attribute_path(output_table_path, 'row_count'))
        if (result_row_count < 0.9 * self.Parameters.magazine_size
        ) and self.Context.node_type == 'map_node' and not self.Context.second_operation_started:
            if result_row_count:
                raw_table_count = int(math.ceil(self.Parameters.magazine_size/float(result_row_count)))
            else:
                raise ValueError("Please, make shure that logging is enabled in your service")
            target_tables = yt.list(canonical_path(self.Context.yt_path), absolute=True)[:raw_table_count]
            yt_operation = self.start_log_processing_operation(target_tables, output_table_path)
            self.Context.second_operation_started = True
            self.Context.yt_operation = yt_operation.id
            logging.info('YT operation %s started', yt_operation.url)

        logging.info('Make resource')
        resorce_service = self.Parameters.replace_service if self.Parameters.replace_service else self.Parameters.service
        resource = sdk2.ResourceData(SaasTankAmmo(
            self, 'SaasTankAmmo', '{}@{}_ammo_{}'.format(self.Parameters.service, self.Parameters.ctype, self.id),
            ttl=14, saas_ctype=self.Parameters.ctype, saas_service=resorce_service
        ))

        rows = yt.read_table(output_table_path)
        counter = 0
        with open(str(resource.path), 'w') as f:
            for row in rows:
                f.write('{}\n'.format(row['ammo']))
                counter += 1
                if counter > 1.3*self.Parameters.magazine_size:
                    break

        resource.ready()
        yt.remove(output_table_path)
