# -*- coding: utf-8 -*-

import logging
import sandbox.sandboxsdk.svn as sdk_svn
import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.sandboxsdk import ssh
from sandbox.projects.saas.DeployFerrymanConfig import DeployFerrymanConfig


ARCADIA_CONFIG_PATH = '/yweb/querydata/querydata_indexer_saas/ferryman/static_configs/data'
LOCAL_CONFIG_PATH_TMPL = 'configs/{}/config-{}.pb.txt'


class ManageFerrymanNamespaces(DeployFerrymanConfig):
    """
    Add and modify namespaces for SaaS Ferryman
    """
    class Parameters(sdk2.Parameters):
        ferryman_configs_dir = sdk2.parameters.ArcadiaUrl(
            'Path to ferryman configs directory',
            required=True,
            default_value=sdk_svn.Arcadia.trunk_url() + ARCADIA_CONFIG_PATH
        )
        ferryman_name = sdk2.parameters.String('Ferryman name', required=True)
        ferryman_ctype = sdk2.parameters.String('Ctype', required=True)
        namespace_name = sdk2.parameters.String('Namespace name', required=True)
        namespace_owners = sdk2.parameters.String('Namespace owners')
        startrek_ticket = sdk2.parameters.String('Startrek ticket')
        ns_doc_count = sdk2.parameters.Integer('Max docs count')
        ns_size = sdk2.parameters.Integer('Size (bytes)')
        dry_run = sdk2.parameters.Bool('Dry run mode', default_value=False)

    def on_prepare(self):
        sdk2.svn.Arcadia.checkout(self.Parameters.ferryman_configs_dir, 'configs')

    def commit(self, message):
        config_path = LOCAL_CONFIG_PATH_TMPL.format(self.Parameters.ferryman_ctype,
                                                    self.Parameters.ferryman_name)
        with ssh.Key(self, 'RTYSERVER-ROBOT', 'ssh_key'):
            return sdk2.svn.Arcadia.commit(config_path, message, user='saas-robot')

    def on_execute(self):
        import saas.tools.ssm.ssm_namespaces as ssm_namespaces
        ssm_namespaces.CONFIG_PATH = LOCAL_CONFIG_PATH_TMPL
        ssm_namespaces_client = ssm_namespaces.FerrymanConfig(self.Parameters.ferryman_name,
                                                              self.Parameters.ferryman_ctype)
        commit_message = ssm_namespaces_client.add_namespace(
            self.Parameters.namespace_name, self.Parameters.ns_doc_count,
            self.Parameters.ns_size,
            ticket=self.Parameters.startrek_ticket, owners=self.Parameters.namespace_owners
        )

        logging.info('Arcadia svn diff\n%s', sdk2.svn.Arcadia.diff('configs'))
        if not self.Parameters.dry_run:
            logging.info('Committing changes.')
            response = self.commit('SKIP_CHECK ' + commit_message)
            rev_id = self.get_revision_id_from_message(response)
            if rev_id:
                logging.info('Revision: ' + rev_id)
            # Get sandbox task
            subtask_id = self.get_id_from_testenv(rev_id)
            # Wait for task before release
            self.get_sandbox_task_instance(subtask_id)
            # Release task
            logging.info('Create release')
            self.server.release(task_id=subtask_id,
                                type=ctt.ReleaseStatus.STABLE,
                                subject='Released task')
            # Deploy
            self.nanny_activate_release(subtask_id)
        else:
            logging.info('Dry run mode was enabled. Changes will not be committed.')
