import logging
import sys
import time
import sandbox.sandboxsdk.svn as sdk_svn
import sandbox.common.types.notification as ctn
from sandbox import sdk2
from sandbox.common.types.task import Status


LOGBROKER_API_DIR = 'arcadia:/arc/trunk/arcadia/saas/rtyserver_test/tests/persqueue'


class RemoveLogbrokerTopics(sdk2.Task):
    """ Remove topics that were created for testing SaaS. """

    class Parameters(sdk2.Parameters):
        owner = sdk2.parameters.String('Owner for logbroker api query', required=True)
        user = sdk2.parameters.String('User for logbroker api query', required=True)
        clusters = sdk2.parameters.String('Clusters separated by commas', required=True)
        consumer = sdk2.parameters.String('Consumer')
        prefix = sdk2.parameters.String('Prefix', required=True)
        ttl = sdk2.parameters.Integer('Topic TTL in seconds', required=True)

        send_email = sdk2.parameters.Bool('Send email', default=False)
        with send_email.value[True]:
            email_receiver = sdk2.parameters.String('Email receiver', required=True)

    def create_api(self):
        sys.path.append(sdk_svn.Arcadia.get_arcadia_src_dir(LOGBROKER_API_DIR))
        token = sdk2.Vault.data('SAAS-ROBOT', 'SAAS_PQ_API_TOKEN')
        import logbroker
        self.api = logbroker.Api(self.Parameters.owner, self.Parameters.user, token)

    def task_is_finished(self, task_id):
        try:
            status = sdk2.Task[task_id].status
        except:
            return False
        return status in tuple(Status.Group.FINISH + Status.Group.BREAK)

    def parse_and_check(self, topic):
        try:
            # rt3.[origin_cluster]--[prefix]-[salt]-[creation_time]-[sb_task_id]-[number_of_topic]--[logtype]
            # rt3.man--saas-testing-9e3c73ff60df4c25ab98bc216b111268-12345678-1509467604-0--raw
            prefix, ident, logtype = topic.split('--')
            _, origin_cluster = prefix.split('.')
            if not ident.startswith(self.Parameters.prefix):
                return False
            _, _, creation_time, task_id, _ = ident.split(self.Parameters.prefix).pop().split('-')
            creation_time = int(creation_time)
        except:
            return False

        topic_age = int(time.time()) - creation_time
        if not self.task_is_finished(task_id) and topic_age < self.Parameters.ttl:
            return False
        return {
            'id': '{ident}--{logtype}'.format(ident=ident, logtype=logtype),
            'origin_cluster': origin_cluster,
            'ident': ident,
            'logtype': logtype,
            'clusters': []
        }

    def delete_topics(self, topics_info, consumer):
        for topic in topics_info:
            info = topics_info[topic]
            self.api.delete_topic(info['origin_cluster'], info['clusters'], info['ident'], info['logtype'], consumer)
        return self.api.run_query('SaaS tests: remove unused topics')

    def send_email(self, removed,  potential, result):
        if not removed and not  potential:
            return
        logging.info('Send email to : {}'.format(self.Parameters.email_receiver))
        subject = 'RemoveLogbrokerTopics: not empty result'
        body = '''
            Results of sandbox task RemoveLogbrokerTopics #{task_id}
            Result of the removal: {result}
            Removed topics {removed_count}:
                {removed_list}
            Potential topics to delete {potential_count}:
                {potential_list}
            '''.format(
                    task_id=self.id,
                    result=result,
                    removed_count=len(removed),
                    removed_list='\n\t'.join(removed) if removed else 'empty',
                    potential_count=len(potential),
                    potential_list='\n\t'.join(potential) if potential else 'empty'
                )

        self.server.notification(
            subject=subject.encode('utf-8'),
            body=body.encode('utf-8'),
            recipients=self.Parameters.email_receiver.split(','),
            transport=ctn.Transport.EMAIL
        )

    def on_execute(self):
        clusters = self.Parameters.clusters.split(',')

        self.create_api()

        selected_topics_info = {}
        selected_topics = []
        potential_topics = []
        for cluster in clusters:
            logging.info('Get topics in cluster={} for {}'.format(cluster, self.Parameters.consumer))
            topics = self.api.get_topic_list(cluster=cluster, consumer=self.Parameters.consumer)
            logging.info('result: ' + ','.join(topics))
            for topic in topics:
                topic_info = self.parse_and_check(topic)
                if topic_info:
                    selected_topics.append('{}:{}'.format(cluster, topic))
                    id = topic_info['id']
                    if not selected_topics_info.get(id):
                        selected_topics_info[id] = topic_info
                    selected_topics_info[id]['clusters'].append(cluster)

            logging.info('Get all topics in cluster={}'.format(cluster))
            topics = self.api.get_topic_list(cluster=cluster)
            logging.info('result: ' + ','.join(topics))
            for topic in topics:
                topic_info = self.parse_and_check(topic)
                if topic_info:
                    id = topic_info['id']
                    if not selected_topics_info.get(id) or cluster not in selected_topics_info[id]['clusters']:
                        potential_topics.append('{}:{}'.format(cluster, topic))

        logging.info('selected:\n\t{}'.format('\n\t'.join(selected_topics)))
        logging.info('potential:\n\t{}'.format('\n\t'.join(potential_topics)))

        result = 'nothing to remove'
        if selected_topics:
            result = self.delete_topics(selected_topics_info, self.Parameters.consumer)
            logging.info('Delete result: {}'.format(result))

        if self.Parameters.send_email:
            self.send_email(selected_topics, potential_topics, result)
