# -*- coding: utf-8 -*-

import json
import tempfile
import sandbox.sdk2.parameters as parameters
from sandbox.projects.gencfg.BaseGencfgGuiRequest import BaseGencfgGuiRequest
import sandbox.projects.saas.common.semaphores as semaphores


class ReplaceHosts(BaseGencfgGuiRequest):
    """Replace hosts in Saas gencfg groups"""

    class Requirements(BaseGencfgGuiRequest.Requirements):
        semaphores = semaphores.SaasSemaphore.gencfg

    class Parameters(BaseGencfgGuiRequest.Parameters):
        with parameters.Group('Request params') as request_params:
            group_name = parameters.String('GenCfg group', required=True)
            hosts_to_replace = parameters.List('Hosts to replace', required=True)

        with parameters.Output:
            changes = parameters.JSON(
                'Changes made by this task', required=True
            )

    def on_enqueue(self):
        if not self.Parameters.author:
            self.Parameters.author = self.author

    def run_subtask_payload(self, gencfg, params):
        self.replace_hosts_in_group(gencfg, params)

    @staticmethod
    def list_to_comma_separated(lst):
        return ','.join(map(str, lst))

    def run_full_checks(self, gencfg, params):
        gencfg.gen_sh('run_checks')

    def replace_hosts_in_group(self, gencfg, params):
        group_name = params.group_name

        changes_file = tempfile.NamedTemporaryFile()
        add_hosts_command = [
            './utils/saas/update_saas_group.py',
            '-g', group_name,
            '-nh', '+{}'.format(len(params.hosts_to_replace)),
            '--file', changes_file.name
        ]
        remove_hosts_str = self.list_to_comma_separated(params.hosts_to_replace)
        remove_hosts_command = [
            './utils/common/update_igroups.py',
            '-y',
            '-a', 'removeslavehosts',
            '-g', group_name,
            '-s', remove_hosts_str
        ]
        gencfg.run_process(add_hosts_command, 'update_group_command')

        with open(changes_file.name) as cf:
            changes = json.load(cf)
            changes[group_name]['removed_hosts'] = remove_hosts_str.split(',')
            self.Parameters.changes = changes
            self.Context.new_hosts = changes[group_name]['added_hosts']

        gencfg.run_process(remove_hosts_command, 'remove_hosts_command')
