# coding: utf-8
import time

import sandbox.sdk2 as sdk2
import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm

from collections import defaultdict
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi

from sandbox.projects.saas.common.classes import SaasBinaryTask, DmProxyHandles
from sandbox.projects.saas.MakeAmmoFromYtLogs import MakeAmmoFromYtLogs, SaasTankAmmo
from sandbox.projects.saas.StoreShootingMetrics import StoreShootingMetrics
from sandbox.projects.saas.StoreShootingRequestsInfo import StoreShootingRequestsInfo


class ShootToSaasService(SaasBinaryTask):
    """Load test SaaS service"""

    TASKS_RESOURCE_NAME = 'SaasLoadTestTasks'
    DISABLED_PROXY_LABEL = 'disable_search'

    class Parameters(sdk2.Task.Parameters):
        ctype = sdk2.parameters.String('SaaS ctype', required=True, hint=True)
        service = sdk2.parameters.String('SaaS service name', required=True, hint=True)
        source_service = sdk2.parameters.String('SaaS service the donor of ammo', required=False, default_value='')

        ammo_file = sdk2.parameters.Resource('Ammo')
        use_existing_ammo = sdk2.parameters.Bool('Use existing ammo from old tasks', default_value=True)
        target_rps = sdk2.parameters.Integer('Target RPS per location (override SLA RPS from DM)')
        ticket = sdk2.parameters.String('Startrek ticket id', required=True)
        tanks = sdk2.parameters.List(
            'Tanks endpoint sets', required=True,
            default=['man@saas_tanks.division_1', 'sas@saas_tanks.division_1', 'vla@saas_tanks.division_1']
        )
        responsibles = sdk2.parameters.List(
            'Responsibles for service to notify', default=[])
        notify_time = sdk2.parameters.Integer('Time to notify responsibles')

        with sdk2.parameters.CheckGroup('Shoot target locations') as locations:
            locations.values.man = locations.Value('MAN', checked=True)
            locations.values.sas = 'SAS'
            locations.values.vla = 'VLA'

        dry_run = sdk2.parameters.Bool('Force shoot RPS==1', default_value=False)
        send_results_to_uchenki = sdk2.parameters.Bool('Send results to uchenki', default_value=False)

        with sdk2.parameters.Group('Requests info') as requests_info:
            store_requests_info = sdk2.parameters.Bool(
                'Store failed requests info to YT & Startrek',
                default_value=False)
            requests_info_find_by = sdk2.parameters.String(
                'CGI ({key}={value}) expression for searching relevant rows in YT requests\' log tables '
                '(may be generated by MAKE_AMMO_FROM_YT_LOGS ammo task, if it\'s not set explicitly). '
                'WARNING: keep in mind that all the matches are searched by YQL LIKE',
                default_value=None,
            )

        with sdk2.parameters.Group('Authorization') as authorization:
            yt_secret_id = sdk2.parameters.String('YAV secret id with YT token',
                                                  default_value='sec-01d74p6hkcabgfq4cv38mqyn9k')
            yt_secret_key = sdk2.parameters.String('YAV secret key for YT', default_value='yt_token')

            yql_secret_id = sdk2.parameters.String('YAV secret id with YQL token',
                                                   default_value='sec-01dmzfn3qhxes8y4yyfar0xyhs')
            yql_secret_key = sdk2.parameters.String('YAV secret key for YQL', default_value='yql_token')

            startrek_secret_id = sdk2.parameters.String('YAV secret id with Startrek token',
                                                        default_value='sec-01farbtgc1wemfrbd5k8wptftw')
            startrek_secret_key = sdk2.parameters.String('YAV secret key for Startrek', default_value='startrek_token')

            warden_secret_id = sdk2.parameters.String('YAV secret id with warden token',
                                                      default_value='sec-01faspfey2x09wds5neabng4vb')
            warden_secret_key = sdk2.parameters.String('YAV secret key for warden', default_value='warden_token')

            tvm_secret_id = sdk2.parameters.String('YAV secret id with TVM token',
                                                   default_value='sec-01ehywawrh8mepfexm5mq2dg55')
            tvm_secret_key = sdk2.parameters.String('YAV secret key for TVM', default_value='client_secret')

    def client_name(self):
        return 'SB:SHOOT_TO_SAAS_SERVICE:{}'.format(self.id)

    def notify_responsibles(self):
        if not self.Parameters.responsibles:
            return

        import logging
        from startrek_client import Startrek

        notify_hours = int(self.Parameters.notify_time / 3600)
        suffix = 'час'
        if notify_hours % 10 > 1 and notify_hours % 10 < 5:
            suffix = 'часа'
        elif notify_hours % 10 > 5 and notify_hours % 10 < 9 or notify_hours % 10 == 0:
            suffix = 'часов'

        soon_start_comment = 'Сервис {} в ctype {} будет обстрелян через {} {}, ' \
                             'отменить это остановив задачу по обстрелу https://sandbox.yandex-team.ru/task/{}'

        MAIN_TICKET = self.Parameters.ticket
        startrek_token_secret = sdk2.yav.Secret(self.Parameters.startrek_secret_id)
        startrek_token = startrek_token_secret.data()[self.Parameters.startrek_secret_key]
        client = Startrek(api_version='v2', useragent='saas-robot', token=startrek_token)

        logging.info(str(client.issues[MAIN_TICKET]))

        if self.Parameters.responsibles:
            client.issues[MAIN_TICKET].comments.create(
                text=soon_start_comment.format(self.Parameters.service, self.Parameters.ctype, notify_hours,
                                               suffix, self.id), summonees=self.Parameters.responsibles)
        else:
            client.issues[MAIN_TICKET].comments.create(
                text=soon_start_comment.format(self.Parameters.service, self.Parameters.ctype, notify_hours, suffix,
                                               self.id))
        return

    def get_tanks(self, geo):
        if self.Context.tanks == ctm.NotExists:
            resolve = self.resole_endpoint_sets(self.Parameters.tanks)
            tanks = defaultdict(list)
            for geo, endpoints in resolve:
                tanks[geo].append(endpoints.vlaues())
            self.Context.tanks = tanks
        return self.Context.tanks[geo]

    def ensure_ammo(self):
        self.Context.ammo_resource = None if not self.Parameters.ammo_file else self.Parameters.ammo_file.id
        if not self.Context.ammo_resource and self.Parameters.use_existing_ammo:
            ammo_resource = SaasTankAmmo.find(state='READY', attrs=dict(
                saas_ctype=self.Parameters.ctype,
                saas_service=self.Parameters.service
            )).first()
            self.Context.ammo_resource = None if not ammo_resource else ammo_resource.id
        if not self.Context.ammo_resource:
            with self.memoize_stage.generate_ammo:
                if self.Parameters.source_service == '':
                    service = self.Parameters.service
                    replace_service = ''
                else:
                    service = self.Parameters.source_service
                    replace_service = self.Parameters.service

                subtask = MakeAmmoFromYtLogs(
                    self,
                    description='Generate ammo',
                    ctype=self.Parameters.ctype,
                    service=service,
                    replace_service=replace_service,
                    magazine_size=int(self.Context.target_rps) * 120,
                ).enqueue()

                self.Context.requests_info_find_by = 'reqinfo=saas-ammo-task-{task_id}'.format(task_id=subtask.id)
                self.Context.subtask = subtask.id

                raise sdk2.WaitTask([subtask], ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=False)

            subtask = MakeAmmoFromYtLogs.find(id=self.Context.subtask).first()
            self.Context.ammo_resource = SaasTankAmmo.find(task=subtask).first().id

    def is_service_tvm(self):
        import re
        from saas.library.python.deploy_manager_api.client import DeployManagerApiClient

        searchproxy_file = DeployManagerApiClient()._get_dm_url_json(
            url='process_storage',
            params={'path': '/configs/{service}/searchproxy-{service}.conf'.format(service=self.Parameters.service)}
        )
        searchproxy_file = str(searchproxy_file['data']).replace(' ', '').replace('\n', '')
        tvm_string = re.search(r'<Tvm>.*</Tvm>', searchproxy_file)
        if not tvm_string:
            return False

        tvm_string = tvm_string.group(0)
        mode_string = re.search(r'Mode:\w+', tvm_string)
        if not mode_string:
            return False

        mode_string = mode_string.group(0)
        if mode_string.split(':')[-1] == 'Enabled':
            return True

        return False

    def collect_allowed_tvms(self):
        import re
        from saas.library.python.deploy_manager_api.client import DeployManagerApiClient
        searchproxy_file = DeployManagerApiClient()._get_dm_url_json(url='process_storage', params={
            'path': '/configs/{service}/searchproxy-{service}.conf'.format(service=self.Parameters.service)})
        searchproxy_file = str(searchproxy_file['data']).replace(' ', '').replace('\n', '')
        allowed_string = re.search(r'<AllowedSourceServiceIds>.*</AllowedSourceServiceIds>', searchproxy_file).group(0)
        allowed_str_by_ctype = re.findall(r'\w+:[\d+,*]+', allowed_string)
        allowed_by_ctype = {}
        for el in allowed_str_by_ctype:
            key, value_str = el.split(':')
            allowed_by_ctype[key] = value_str.split(',')
        return set([int(el) for el in allowed_by_ctype[self.Parameters.ctype]])

    def collect_searchproxy_tvms(self):
        import re
        from saas.library.python.deploy_manager_api.client import DeployManagerApiClient

        environment_file = DeployManagerApiClient()._get_dm_url_json(url='process_storage', params={
            'path': '/common/{}/environment'.format(self.Parameters.ctype)})
        environment_file = str(environment_file['data']).replace(' ', '').replace('\n', '')
        service_ids = re.search(r'TVM_SERVICE_IDS=[\d+,*]+', environment_file).group(0).split('=')[-1].split(',')
        service_id = re.search(r'TVM_SERVICE_ID=\d+', environment_file).group(0).split('=')[-1]
        service_ids.append(service_id)
        return set([int(el) for el in service_ids])

    def get_tvm_ticket(self):
        import logging
        from tvmauth import TvmClient, TvmApiClientSettings

        self.Context.source_tvm_app = 2023414
        allowed_tvms = self.collect_allowed_tvms()
        searchproxy_tvms = self.collect_searchproxy_tvms()

        class TvmAuthError(Exception):
            pass

        if self.Context.source_tvm_app not in allowed_tvms:
            raise TvmAuthError('Please, make sure that you have all tvm accesses')

        tvm_token_secret = sdk2.yav.Secret(self.Parameters.tvm_secret_id)
        tvm_token = tvm_token_secret.data()[self.Parameters.tvm_secret_key]
        tvm_settings = TvmApiClientSettings(self_secret=tvm_token, self_tvm_id=self.Context.source_tvm_app,
                                            dsts={'service': int(list(searchproxy_tvms)[0])})
        tvm_client = TvmClient(settings=tvm_settings)
        tvm_ticket = tvm_client.get_service_ticket_for('service')
        logging.info('Got TVM ticket for shooting: {}***{}'.format(tvm_ticket[0], tvm_ticket[-1]))
        return tvm_ticket

    def shoot(self, geo):
        import logging

        from sandbox.projects.saas.common.shooting import SingleShootingResult

        from saas.library.python.yasm import SaasServiceMetrics
        from saas.library.python.tank import build_tank_config
        from saas.library.python.yp_service_discovery import Resolver

        resolver = Resolver(self.client_name())

        tanks_endpoints = resolver.resolve_endpoint_set(geo, self.Context.tanks_endpoints[geo])
        tanks = [t.fqdn for t in tanks_endpoints]

        proxy_endpoints = resolver.resolve_endpoint_set(
            geo, self.Context.proxy_endpoints[geo], label_selectors=[self.DISABLED_PROXY_LABEL]
        )
        proxies = [p.fqdn for p in proxy_endpoints if p.labels[self.DISABLED_PROXY_LABEL] != 'True']

        yasm = SaasServiceMetrics(self.Parameters.ctype, self.Parameters.service)
        current_rps = int(yasm.proxy_rps(geo=geo).last_thirty_minutes.mean(normalize=True))

        if self.is_service_tvm():
            tvm_ticket = self.get_tvm_ticket()
            headers = ['X-Ya-Service-Ticket: {}'.format(tvm_ticket)]
        else:
            headers = None

        logging.info('Current rps: %d', current_rps)

        if current_rps >= self.Context.target_rps:
            self.Context.final_measurements['results'].append(
                SingleShootingResult(geo=geo, rps=current_rps, target=self.Context.target_rps).asdict())
            logging.fatal(
                'Current RPS={current_rps} in {geo} is over target RPS={target_rps}. Skip geo'.format(
                    current_rps=current_rps, geo=geo, target_rps=self.Context.target_rps
                )
            )
            raise sdk2.WaitTime(1)
        else:
            shoot_rps = int(self.Context.target_rps - current_rps)
            tank_config = build_tank_config(
                ctype=self.Parameters.ctype,
                service=self.Parameters.service,
                targets=proxies,
                target_rps=shoot_rps,
                ticket=self.Parameters.ticket,
                user=self.author,
                headers=headers
            )
            subtask = ShootViaTankapi(
                self,
                tanks=tanks,
                description='Load Test SaaS service {}@{}'.format(self.Parameters.service, self.Parameters.ctype),
                use_public_tanks=False,
                config_source='file',
                config_content=tank_config,
                ammo_source='resource',
                ammo_resource=self.Context.ammo_resource
            ).enqueue()
            self.Context.final_measurements['results'].append(
                SingleShootingResult(geo=geo, rps=current_rps, target=self.Context.target_rps,
                                     task_id=subtask.id).asdict())
            raise sdk2.WaitTask([subtask], ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=False)

    def on_execute(self):
        import logging
        from saas.library.python.deploy_manager_api import SaasService
        from sandbox.projects.saas.common.shooting import ShootingResults

        if self.Parameters.notify_time:
            with self.memoize_stage.notify:
                self.notify_responsibles()
                raise sdk2.WaitTime(self.Parameters.notify_time)

        if not self.Context.target_rps:
            if self.Parameters.target_rps:
                self.Context.target_rps = self.Parameters.target_rps
            else:
                dm_proxy = DmProxyHandles()
                sla_rps = dm_proxy.get_sla_rps(self.Parameters.ctype, self.Parameters.service)
                saas_service = SaasService(self.Parameters.ctype, self.Parameters.service)
                basic_locations_num = len(set(saas_service.locations).intersection({'MAN', 'SAS', 'VLA'}))
                self.Context.target_rps = int(sla_rps / (1 if basic_locations_num == 1 else (basic_locations_num - 1)))

        startrek_secret = sdk2.yav.Secret(self.Parameters.startrek_secret_id)
        startrek_token = startrek_secret.data()[self.Parameters.startrek_secret_key]

        warden_secret = sdk2.yav.Secret(self.Parameters.warden_secret_id)
        warden_token = warden_secret.data()[self.Parameters.warden_secret_key]

        yt_secret = sdk2.yav.Secret(self.Parameters.yt_secret_id)
        yt_token = yt_secret.data()[self.Parameters.yt_secret_key]

        ShootingResults.init_yt_client(yt_token)
        ShootingResults.init_startrek_client(startrek_token)
        ShootingResults.init_warden_client(warden_token)
        logging.info('Target rps: %d', self.Context.target_rps)
        self.ensure_ammo()
        logging.debug('Ammo acquired: %s', self.Context.ammo_resource)

        with self.memoize_stage.prepare:
            from saas.library.python.deploy_manager_api import DeployManagerApiClient
            from saas.library.python.token_store import PersistentTokenStore

            PersistentTokenStore.add_token('nanny', '')

            self.Context.locations_count = len(self.Parameters.locations)
            self.Context.target_locations = self.Parameters.locations

            self.Context.tanks_endpoints = dict([t.split('@') for t in self.Parameters.tanks])
            self.Context.proxy_endpoints = dict(
                [p.split('@') for p in DeployManagerApiClient().get_proxy_endpoint_sets(self.Parameters.ctype)])
            self.Context.target_locations = self.Parameters.locations
            self.Context.final_measurements = ShootingResults(task_id=self.id,
                                                              service=self.Parameters.service,
                                                              ctype=self.Parameters.ctype).asdict()
            self.Context.shooting_start_ts = int(time.time())

        with self.memoize_stage.shoot(self.Context.locations_count):
            target_location = self.Context.target_locations.pop()
            self.shoot(target_location)

        if not self.Context.final_measurements['results']:
            return

        self.Context.shooting_end_ts = int(time.time())

        with self.memoize_stage.threat_results:
            from saas.library.python.yasm.saas_service_metrics import SaasServiceMetrics
            metrics = SaasServiceMetrics(self.Parameters.ctype, self.Parameters.service)
            for i in range(len(self.Context.final_measurements['results'])):
                results = self.Context.final_measurements['results']
                shtg = ShootingResults.fromdict(self.Context.final_measurements).results[i]
                time_range = shtg.lunapark_job.time_range
                if time_range is not None:
                    results[i]['rps'] = metrics.proxy_rps(geo=shtg.geo).max(*time_range, normalize=True)

        with self.memoize_stage.fill_ticket_and_info:
            self.Context.ticket_id = ShootingResults.fromdict(self.Context.final_measurements).generate_ticket(
                self.Parameters.ticket
            )
            self.set_info(
                ShootingResults.fromdict(self.Context.final_measurements).reduce_information_for_sandbox(),
                do_escape=False
            )

        self._store_shooting_metrics()
        self._send_results_to_uchenki_if_needed()
        self._store_shooting_requests_info_if_needed()

    def _store_shooting_metrics(self):
        with self.memoize_stage.store_shooting_metrics:
            subtask = StoreShootingMetrics(
                self,
                description='Upload shooting metrics to YT',
                shooting_task_id=self.id,
                yt_secret_id=self.Parameters.yt_secret_id,
                yt_secret_key=self.Parameters.yt_secret_key,
            ).enqueue()
            raise sdk2.WaitTask([subtask], ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=False)

    def _send_results_to_uchenki_if_needed(self):
        if not self.Parameters.send_results_to_uchenki:
            return

        from sandbox.projects.saas.common.shooting import ShootingResults

        with self.memoize_stage.get_or_create_component:
            ShootingResults.fromdict(self.Context.final_measurements).get_or_create_component()
        ShootingResults.fromdict(self.Context.final_measurements).send_to_uchenki(self.Context.ticket_id)

    def _store_shooting_requests_info_if_needed(self):
        if not self.Parameters.store_requests_info:
            return

        with self.memoize_stage.store_shooting_requests_info:
            subtask = StoreShootingRequestsInfo(
                self,
                description='Upload shooting requests info to YT & Startrek',

                shooting_task_id=self.id,
                startrek_ticket_id=self.Context.ticket_id,
                find_by=self.Parameters.requests_info_find_by or self.Context.requests_info_find_by,

                yt_secret_id=self.Parameters.yt_secret_id,
                yt_secret_key=self.Parameters.yt_secret_key,
                yql_secret_id=self.Parameters.yql_secret_id,
                yql_secret_key=self.Parameters.yql_secret_key,
                startrek_secret_id=self.Parameters.startrek_secret_id,
                startrek_secret_key=self.Parameters.startrek_secret_key,
            ).enqueue()
            raise sdk2.WaitTask([subtask], ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=False)
