# coding: utf-8
import re
import sandbox.sdk2 as sdk2

from sandbox.projects.saas.common.classes import SaasBinaryTask


class StoreShootingRequestsInfo(SaasBinaryTask):

    """Load test SaaS service"""
    TASKS_RESOURCE_NAME = 'SaasLoadTestTasks'
    WAIT_TIME = 30 * 60  # 30m

    class Parameters(sdk2.Task.Parameters):
        shooting_task_id = sdk2.parameters.String('Shooting task id', required=True)
        startrek_ticket_id = sdk2.parameters.String('Startrek ticket id', required=True)

        find_by = sdk2.parameters.String(
            'CGI ({key}={value}) expression for searching relevant rows in YT requests\' log tables '
            '(may be generated by MAKE_AMMO_FROM_YT_LOGS ammo task, if it\'s not set explicitly). '
            'WARNING: keep in mind that all the matches are searched by YQL LIKE',
            default_value=None,
        )

        with sdk2.parameters.Group('Authorization') as authorization:
            yt_secret_id = sdk2.parameters.String(
                'YAV secret id with YT token', default_value='sec-01d74p6hkcabgfq4cv38mqyn9k')
            yt_secret_key = sdk2.parameters.String('YAV secret key for YT', default_value='yt_token')

            yql_secret_id = sdk2.parameters.String(
                'YAV secret id with YQL token', default_value='sec-01dmzfn3qhxes8y4yyfar0xyhs')
            yql_secret_key = sdk2.parameters.String('YAV secret key for YQL', default_value='yql_token')

            startrek_secret_id = sdk2.parameters.String('YAV secret id with Startrek token',
                                                        default_value='sec-01farbtgc1wemfrbd5k8wptftw')
            startrek_secret_key = sdk2.parameters.String('YAV secret key for Startrek', default_value='startrek_token')

    def _validate_options(self):
        find_by = self.Parameters.find_by
        pattern = re.compile(r'^[\w-]+=[\w-]+$')
        if not pattern.match(find_by):  # noqa
            raise ValueError('Parameter `find_by` should be in format: `{key}={value}`, e.g. reqinfo=saas-shooting')

    def on_execute(self):
        self._validate_options()

        from sandbox.projects.saas.common.shooting import ShootingResults

        yt_secret = sdk2.yav.Secret(self.Parameters.yt_secret_id)
        yt_token = yt_secret.data()[self.Parameters.yt_secret_key]
        ShootingResults.init_yt_client(yt_token)

        yql_secret = sdk2.yav.Secret(self.Parameters.yql_secret_id)
        yql_token = yql_secret.data()[self.Parameters.yql_secret_key]
        ShootingResults.init_yql_client(yql_token)

        startrek_secret = sdk2.yav.Secret(self.Parameters.startrek_secret_id)
        startrek_token = startrek_secret.data()[self.Parameters.startrek_secret_key]
        ShootingResults.init_startrek_client(startrek_token)

        t = sdk2.Task[self.Parameters.shooting_task_id]
        m = t.Context.final_measurements
        result = ShootingResults.fromdict(m).store_requests_info(
            find_by=self.Parameters.find_by,
            shooting_start_ts=t.Context.shooting_start_ts,
            shooting_end_ts=t.Context.shooting_end_ts,
            startrek_ticket_id=self.Parameters.startrek_ticket_id,
        )
        if not result:
            raise sdk2.WaitTime(self.WAIT_TIME)
