import logging
import os
import requests
import sandbox.sandboxsdk.svn as sdk_svn
from sandbox import sdk2

DEFAULT_SAAS_REFRESH_ALERTS_DIR = '/saas/tools/devops/refresh_alerts'


class TAlertTemplate:
    COMMNAD_URLS = {
        'check': 'https://yasm.yandex-team.ru/srvambry/tmpl/alerts/render_json/%s',
        'create': 'https://yasm.yandex-team.ru/srvambry/tmpl/alerts/create?key=%s',
        'update': 'https://yasm.yandex-team.ru/srvambry/tmpl/alerts/update/content?key=%s',
        'delete': 'https://yasm.yandex-team.ru/srvambry/tmpl/alerts/delete?key=%s',
        'apply': 'https://yasm.yandex-team.ru/srvambry/tmpl/alerts/apply/%s'
    }

    def __init__(self, owner, key, dir=None, macros=False):
        self.macros = macros
        self.data = {
            'key': key,
            'owners': [owner]
        }
        if dir is not None:
            filename = os.path.join(dir, key)
            with open(filename, 'r') as template_file:
                self.data['content'] = template_file.read()

    def __execute(self, url, json=None, data=None, retries=3):
        url = url % self.data.get('key')
        logging.info('request: {}'.format(url))
        for _ in range(retries):
            try:
                r = requests.post(
                    url,
                    json=json,
                    data=data,
                    timeout=1200
                )
                break
            except Exception as e:
                logging.error('request failed: {}'.format(e))
        else:
            logging.error('request failed for {} retries'.format(retries))
            return False

        logging.info(r.reason)
        logging.debug(r.content)
        return r.status_code == requests.codes.ok

    def run(self, command):
        if self.macros and command in ['check', 'apply']:
            logging.info('{}: skip {}, because it is macros'.format(
                self.data.get('key'),
                command
            ))
            return True
        if command in ['update', 'check']:
            return self.__execute(self.COMMNAD_URLS[command], data=self.data.get('content'))
        return self.__execute(self.COMMNAD_URLS[command], json=self.data)


class UpdateSaasRefreshAlerts(sdk2.Task):
    """ Update SaaS.Refresh alerts from teplates. """

    class Parameters(sdk2.Parameters):
        alerts_owner = sdk2.parameters.String('Owner', required=True)

        with sdk2.parameters.RadioGroup("Action") as action:
            action.values['create'] = action.Value(value='create', default=True)
            action.values['delete'] = action.Value(value='delete')
            action.values['update'] = action.Value(value='update')

        with action.value['create'], action.value['update']:
            arcadia_templates_dir = sdk2.parameters.ArcadiaUrl(
                'Path to the templates',
                required=True,
                default_value=sdk_svn.Arcadia.trunk_url() + DEFAULT_SAAS_REFRESH_ALERTS_DIR
            )

        with action.value['create'], action.value['delete']:
            templates_list = sdk2.parameters.List('Templates list', required=True)

        with action.value['update']:
           allow_creation = sdk2.parameters.Bool('Allow creation of new templates', default=False)

    def load_templates(self, local_dir, files=None):
        import re
        template_regex = re.compile('^[\w_]+$')
        macros_regex = re.compile('^[\w_]+\.macros$')
        files_in_dir = os.listdir(local_dir)
        logging.info('List of all files in directory: {}'.format(files_in_dir))
        macros_files = []
        template_files = []
        if files is None:
            template_files = filter(lambda f: re.search(template_regex, f), files_in_dir)
            macros_files = filter(lambda f: re.search(macros_regex, f), files_in_dir)
        else:
            for filename in files:
                if re.search(template_regex, filename):
                    template_files.append(filename)
                elif re.search(macros_regex, filename):
                    macros_files.append(filename)
                else:
                    raise Exception('Incorrect filename in templates list: {}'.format(filename))
        logging.info('Load templates: {}'.format(template_files))
        logging.info('Load macros: {}'.format(macros_files))
        if not template_files and not macros_files:
            raise Exception('Empty lists of files')
        templates = []
        for filename in macros_files:
            templates.append(TAlertTemplate(self.Parameters.alerts_owner, filename, dir=local_dir, macros=True))
        for filename in template_files:
            templates.append(TAlertTemplate(self.Parameters.alerts_owner, filename, dir=local_dir))
        return templates

    def get_alert_templates(self, files=None):
        arcadia_local_dir = sdk_svn.Arcadia.get_arcadia_src_dir(self.Parameters.arcadia_templates_dir)
        return self.load_templates(arcadia_local_dir, files)

    def run(self, templates, command):
        logging.info('Run command={} for templates...'.format(command))
        for template in templates:
            if not template.run(command):
                if command == 'update' and self.Parameters.allow_creation:
                    if template.run('create'):
                        logging.info('Created instead of updating key={}'.format(template.data.get('key')))
                        continue
                raise Exception('Cannot run command={} for key={}'.format(command, template.data.get('key')))

    def on_execute(self):
        logging.info(self.Parameters.action)
        if self.Parameters.action == 'create':
            logging.info('Create templates...')
            templates = self.get_alert_templates(self.Parameters.templates_list)
            self.run(templates, 'create')
            self.run(templates, 'apply')
        elif self.Parameters.action == 'delete':
            logging.info('Delete templates...')
            templates = [TAlertTemplate(self.Parameters.alerts_owner, key) for key in self.Parameters.templates_list]
            self.run(templates, 'delete')
        elif self.Parameters.action == 'update':
            logging.info('Update templates...')
            templates = self.get_alert_templates()
            self.run(templates, 'check')
            self.run(templates, 'update')
            self.run(templates, 'apply')
        else:
            raise Exception('Incorrect input action: {}'.format(self.Parameters.action))
