# -*- coding: utf-8 -*-

import json
import tempfile
import textwrap
import sandbox.sdk2.parameters as parameters

from sandbox.projects.gencfg.BaseGencfgGuiRequest import BaseGencfgGuiRequest
import sandbox.projects.saas.common.semaphores as semaphores


class UpdateSaasGroup(BaseGencfgGuiRequest):
    """Make some changes in SaaS gencfg groups"""

    class Requirements(BaseGencfgGuiRequest.Requirements):
        semaphores = semaphores.SaasSemaphore.gencfg

    class Parameters(BaseGencfgGuiRequest.Parameters):
        with parameters.Group('Request params') as request_params:
            group_name = parameters.List('GenCfg groups', required=True)
            instance_count = parameters.StrictString('Change instance number: +x to add; -x to reduce; =x to set exact number (per location)',
                                                     regexp=r'^([+\-\=][1-9]\d*)?$', required=False)
            cpu_requirements = parameters.Integer('Target CPU power in Kimkim Pu (per instance)', required=False)
            cpu_limit = parameters.Integer('Instance CPU limit in Kimkim Pu', required=False)
            mem_requirements = parameters.Integer('Target RAM in Gigabytes (per instance)', required=False)
            max_instances_per_switch = parameters.Integer('Maximum number of instances per switch', required=False)
            exclude_groups = parameters.List('Exclude hosts present in Groups')
            allow_partial_update = parameters.Bool('Allow partial update', description='Change as much overcommitted hosts as possible',
                                                   required=True, default=False)
            with parameters.RadioGroup('No indexing') as no_indexing:
                no_indexing.values.keep = no_indexing.Value(value='keep', default=True)
                no_indexing.values.true = no_indexing.Value(value='true')
                no_indexing.values.false = no_indexing.Value(value='false')
            with parameters.RadioGroup('Network critical') as network_critical:
                network_critical.values.keep = no_indexing.Value(value='keep', default=True)
                network_critical.values.true = no_indexing.Value(value='true')
                network_critical.values.false = no_indexing.Value(value='false')

            io_limits = parameters.JSON('IO limits', required=False,
                                        description=textwrap.dedent("""
                                        Example: '{"ssd": {"read": 25, "write": 20}, "hdd": {"read": 10, "write": 10}}'
                                        """))
            volumes = parameters.JSON('Volumes', required=False,
                                      description=textwrap.dedent("""
                                      volumes update in json: keys are mountpoints, ex:
                                      {"/db/bsconfig/webstate": {"quota": "550 Gb","host_mp_root": "/ssd"}, "/ssd": {"host_mp_root": "/hdd"}, "/cores": {"symlinks": ["/valhalla"]}}
                                      """))

    def on_enqueue(self):
        if not self.Parameters.author:
            self.Parameters.author = self.author

    def run_subtask_payload(self, gencfg, params):
        self.update_group(gencfg, params)

    @staticmethod
    def list_to_comma_separated(lst):
        return ','.join(map(str, lst))

    def run_full_checks(self, gencfg, params):
        gencfg.gen_sh('run_checks')

    def update_group(self, gencfg, params):
        changes_file = tempfile.NamedTemporaryFile()
        update_group_command = [
            './utils/saas/update_saas_group.py',
            '-g', self.list_to_comma_separated(params.group_name),
            '--allow-host-change',
            '--force-overcommit-check',
            '--file', changes_file.name
        ]
        if params.instance_count and params.instance_count != 'None':
            update_group_command.extend(['--instance-number', params.instance_count])
        if params.cpu_requirements:
            update_group_command.extend(['--slot-size-cpu', str(params.cpu_requirements)])
        if params.cpu_limit:
            update_group_command.extend(['--slot-cpu-limit', str(params.cpu_limit)])
        if params.mem_requirements:
            update_group_command.extend(['--slot-size-mem', str(params.mem_requirements)])
        if params.exclude_groups:
            update_group_command.extend(['--blacklisted-groups', self.list_to_comma_separated(params.exclude_groups)])
        if params.no_indexing in ('true', 'false'):
            update_group_command.extend(['--no-indexing', params.no_indexing])
        if params.network_critical in ('true', 'false'):
            update_group_command.extend((['--network-critical', params.network_critical]))
        if params.max_instances_per_switch:
            update_group_command.extend(['--max-instances-per-switch', str(params.max_instances_per_switch)])
        if params.io_limits:
            update_group_command.extend(['--io-limits', json.dumps(params.io_limits)])
        if params.volumes:
            update_group_command.extend(['--disk-volumes', json.dumps(params.volumes)])
        if params.allow_partial_update:
            update_group_command.append('--allow-partial-update')

        gencfg.run_process(update_group_command, 'update_group_command')
        with open(changes_file.name) as cf:
            self.Context.changes = cf.read()
