import logging
import os
from sandbox.projects.common.requests_wrapper import get_r
from sandbox.projects.saas.common.classes import SaasBinaryTask
from sandbox import sdk2

DEFAULT_SAAS_SEARCHMAPS_DIR = '/saas/dm_configs'
DM_ADDRESS = 'http://saas-dm-proxy.n.yandex-team.ru'


def svn_safe_add(path):
    status = sdk2.svn.Arcadia.status(path)
    for status_line in status.splitlines():
        if status_line.startswith('?'):
            new_item = status_line.split().pop()
            sdk2.svn.Arcadia.add(new_item)


class UpdateSaasSearchmap(SaasBinaryTask):
    """ Update SaaS SearchMap from DM to arcadia """
    TASKS_RESOURCE_NAME = 'UpdateSaasSearchmap'
    TASKS_RESOURCE_OWNER = 'RTYSERVER-ROBOT'

    class Parameters(sdk2.Parameters):
        ctypes = sdk2.parameters.String('Ctypes', default='')
        arcadia_searchmaps_dir = sdk2.parameters.ArcadiaUrl(
            'Path to the searchmap without ctype',
            required=True,
            default_value=sdk2.vcs.svn.Arcadia.trunk_url() + DEFAULT_SAAS_SEARCHMAPS_DIR
        )
        update_in_mds = sdk2.parameters.Bool("Update configs in MDS")
        with update_in_mds.value[True]:
            with sdk2.parameters.Group("MDS parameters"):
                mds_access_key_id = sdk2.parameters.String("Vault item name for access key id")
                mds_secret_access_key = sdk2.parameters.String("Vault item name for secret access key")
                mds_production = sdk2.parameters.Bool("Use MDS production")

    def _create_mds_connection(self):
        if not self.Parameters.update_in_mds:
            return None
        from boto.s3.connection import S3Connection
        s3_access_key_id = sdk2.Vault.data(self.owner, self.Parameters.mds_access_key_id)
        s3_access_key = sdk2.Vault.data(self.Parameters.mds_secret_access_key)
        server = 's3.mds.yandex.net' if self.Parameters.mds_production else 's3.mdst.yandex.net'
        return S3Connection(
            host=server,
            aws_access_key_id=s3_access_key_id,
            aws_secret_access_key=s3_access_key)

    def get_config(self, ctype, service, config_name):
        list_conf_url = '{dm_address}/list_conf?service_type=indexerproxy&ctype={ctype}&service={service}'
        get_conf_url = '{dm_address}/get_conf?filename={filename}'
        list_conf = get_r(
            list_conf_url.format(dm_address=DM_ADDRESS, service=service, ctype=ctype),
            timeout=90
        )
        get_config_url = None
        for conf_file in list_conf.json().get('files', []):
            if conf_file.get('rename') == config_name:
                get_config_url = get_conf_url.format(dm_address=DM_ADDRESS, filename=conf_file.get('url'))
                break
        else:
            raise Exception('No "{}" in list_conf responce.'.format(config_name))
        config = get_r(get_config_url, timeout=90).text
        return config

    def init(self):
        self.checkout()
        self.mds_connection = self._create_mds_connection()

    def checkout(self):
        self.local_configs_path = sdk2.svn.Arcadia.checkout(self.Parameters.arcadia_searchmaps_dir, 'configs')
        logging.info('list dir: {}'.format(os.listdir(self.local_configs_path)))

    def commit(self):
        svn_safe_add(self.local_configs_path)
        logging.info('svn status:\n{}'.format(sdk2.svn.Arcadia.status(self.local_configs_path)))
        with sdk2.ssh.Key(self, 'RTYSERVER-ROBOT', 'ssh_key'):
            sdk2.svn.Arcadia.commit(
                self.local_configs_path,
                'update saas config from sandbox #{}: searchmap SKIP_CHECK'.format(self.id), 'saas-robot'
            )

    def update(self):
        ctypes = self.Parameters.ctypes
        if not ctypes:
            from saas.library.python.deploy_manager_api import DeployManagerApiClient
            ctypes = ','.join(DeployManagerApiClient(DM_ADDRESS).ctypes)
            logging.info('automatically discovered ctypes: {}\n'.format(ctypes))
        for ctype in ctypes.split(','):
            self.update_searchmap(ctype)

    def update_searchmap(self, ctype):
        common_service = 'indexerproxy'  # common service for all ctypes
        self.update_config(ctype, common_service, 'searchmap.json')

    def update_config(self, ctype, service, config_name):
        config = self.get_config(ctype, service, config_name)
        self.update_in_repository(config, ctype, service, config_name)
        self.update_in_mds(config, ctype, service, config_name)

    def update_in_repository(self, config_data, ctype, service, config_name):
        local_path = os.path.join(self.local_configs_path, ctype)
        if not os.path.exists(local_path):
            os.makedirs(local_path)
        filename = os.path.join(local_path, config_name)
        with open(filename, 'w') as config_file:
            config_file.write(config_data)

    def update_in_mds(self, config_data, ctype, service, config_name):
        if not self.Parameters.update_in_mds:
            return
        bucket = self.mds_connection.get_bucket('saas-searchmap')
        key = bucket.get_key(ctype)
        if key is None:
            key = bucket.new_key(ctype)
        key.set_contents_from_string(config_data)

    def on_execute(self):
        self.init()
        self.update()
        self.commit()
