# coding: utf-8

import sandbox.sdk2 as sdk2
import sandbox.common.types.task as ctt

from sandbox.projects.saas.RenderSaasShardmap import RenderSaasShardmap
from sandbox.projects.saas.common.classes import SaasBinaryTask
import sandbox.common.types.client as ctc


class UpdateSaasServiceShardmap(SaasBinaryTask):
    """Update shardmap for saas service"""
    TASKS_RESOURCE_NAME = 'SaasShardmapTasks'

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(RenderSaasShardmap.Parameters):
        push_tasks_resource = True
        check_shardmaps = sdk2.parameters.Bool('Make shardmap checking before updating', default_value=True)
        activate_snapshot = sdk2.parameters.Bool('Activate snapshot with new shardmap', default_value=False)
        wait_for_service_activate = sdk2.parameters.Bool('Wait for snapshot activating', default_value=False)
        prepare_recipe = sdk2.parameters.String('Recipe for delivery shardmap to instances', required="")
        activate_recipe = sdk2.parameters.String('Recipe for instances restarting', required="common")
        """Use parameters from RenderSaasShardmap"""

    def update_shardmap(self):
        from saas.library.python.nanny_rest.shard_resource import SandboxShardmap
        from saas.library.python.nanny_rest import NannyServiceBase, AllocationType
        from saas.library.python.deploy_manager_api.saas_service import SaasService
        from saas.library.python.shardmap import collect_nanny_service_recipes
        import logging

        yp_shardmap = SandboxShardmap({
            'task_type': 'RENDER_SAAS_SHARDMAP',
            'resource_type': 'SAAS_SHARD_MAP_YP',
            'task_id': self.Context.shardmap_task_id
        })
        gencfg_shardmap = SandboxShardmap({
            'task_type': 'RENDER_SAAS_SHARDMAP',
            'resource_type': 'SAAS_SHARD_MAP_GENCFG',
            'task_id': self.Context.shardmap_task_id
        })
        self.Context.nanny_services = SaasService(self.Parameters.ctype, self.Parameters.service).nanny_services
        for nanny_service_name in self.Context.nanny_services:
            if not self.Context.nanny_services_to_update[nanny_service_name] and self.Parameters.check_shardmaps:
                continue
            nanny_service = NannyServiceBase(nanny_service_name)
            sandbox_shardmap = yp_shardmap if nanny_service.allocation_type == AllocationType.yp_lite else gencfg_shardmap
            with nanny_service.runtime_attrs_transaction('Update shardmap from sandbox task {}'.format(self.id)) as mutable_nanny_service:
                mutable_nanny_service.shard = sandbox_shardmap

        if not self.Parameters.activate_snapshot:
            return

        avalable_activation_recps = [collect_nanny_service_recipes(service)[0] for service in self.Context.nanny_services]
        avalable_preparing_recps = [collect_nanny_service_recipes(service)[0] for service in
                                     self.Context.nanny_services]
        if not self.Parameters.prepare_recipe in avalable_preparing_recps or not self.Parameters.activate_recipe in avalable_activation_recps:
            logging.error("Can't activate snapshot due to selected recipe not stated in service")
            return

        for nanny_service_name in self.Context.nanny_services:
            nanny_service = NannyServiceBase(nanny_service_name)
            nanny_service.activate_current_snapshot(recipe_name=self.Parameters.activate_recipe, prepare_recipe_name=self.Parameters.prepare_recipe)

    def on_execute(self):
        from saas.library.python.token_store import TokenStore
        from saas.library.python import shardmap
        nanny_secret = sdk2.yav.Secret(self.Parameters.nanny_secret_id)
        TokenStore.add_token('nanny', nanny_secret.data()[self.Parameters.nanny_secret_key])
        shards_data = shardmap.build_shards_data(self.Parameters.yt_cluster, self.Parameters.shards_data, self.Parameters.yt_secret_id, self.Parameters.yt_secret_key) if self.Parameters.shards_data else None
        self.Context.nanny_services_to_update =  shardmap.match_rendered_shardmap(self.Parameters.ctype, self.Parameters.service, shards_data)
        if self.Parameters.check_shardmaps and sum(list(self.Context.nanny_services_to_update.values())) == 0:
            return
        with self.memoize_stage.render_shardmap:
            subtask = RenderSaasShardmap(
                self,
                description="SaaS shardmap for {}@{}. Child of {}".format(
                    self.Parameters.service, self.Parameters.ctype, self.id),
                **dict(self.Parameters)
            ).enqueue()
            raise sdk2.WaitTask([subtask], ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=False)

        shardmap_task = self.find(RenderSaasShardmap).first()
        if not shardmap_task:
            raise RuntimeError('RenderSaasShardmap subtask not found')
        if not sdk2.Resource.find(task_id=shardmap_task.id, type="SAAS_SHARD_MAP_YP").count:
            raise RuntimeError('Required resources were not found in RenderSaasShardmap task. Please, make shure that it\'s all right with endpointsets with service')

        self.Context.shardmap_task_id = shardmap_task.id
        self.update_shardmap()
