# coding: utf-8

import sandbox.sdk2 as sdk2
import sandbox.common.types.task as ctt

from sandbox.projects.saas.UpdateSaasServiceShardmap import UpdateSaasServiceShardmap
from sandbox.projects.saas.common.classes import SaasBinaryTask
import sandbox.common.types.client as ctc


class UpdateSaasServicesShardmaps(SaasBinaryTask):
    """Update shardmap for saas service"""
    TASKS_RESOURCE_NAME = 'SaasShardmapTasks'

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):
        push_tasks_resource = True
        with sdk2.parameters.Group('Authorisation') as authorisation:
            nanny_secret_id = sdk2.parameters.String('YAV secret id with Nanny token',
                                                     default_value='sec-01dmzfn3qhxes8y4yyfar0xyhs')
            nanny_secret_key = sdk2.parameters.String('YAV secret key for Nanny',
                                                      default_value='nanny_token')
            yt_secret_id = sdk2.parameters.String('YAV secret id with YT token',
                                                  default_value='sec-01d74p6hkcabgfq4cv38mqyn9k')
            yt_secret_key = sdk2.parameters.String('YAV secret key for YT', default_value='yt_token')

    def on_execute(self):
        from saas.library.python.deploy_manager_api import helpers
        from saas.library.python.shardmap import shardmap
        from saas.library.python.token_store import TokenStore
        from nanny_rpc_client.exceptions import InternalError, NotFoundError, ForbiddenError
        import logging

        nanny_secret = sdk2.yav.Secret(self.Parameters.nanny_secret_id)
        TokenStore.add_token('nanny', nanny_secret.data()[self.Parameters.nanny_secret_key])

        saas_services = helpers.saas_service_iterator()
        subtasks = list()
        with self.memoize_stage.update_saas_shardmaps:
            for saas_service in saas_services:
                try:
                    nanny_services_to_update = shardmap.match_rendered_shardmap(saas_service.ctype, saas_service.name, None)
                except InternalError as e:
                    logging.warning("Exception {} with service {}@{}".format(str(e), saas_service.name, saas_service.ctype))
                    continue
                except NotFoundError as e:
                    logging.warning("Exception {} with service {}@{}".format(str(e), saas_service.name, saas_service.ctype))
                    continue
                except ForbiddenError as e:
                    logging.warning("Exception {} with service {}@{}".format(str(e), saas_service.name, saas_service.ctype))
                    continue
                if sum(list(nanny_services_to_update.values())) == 0:
                    continue
                subtask = UpdateSaasServiceShardmap(self, description="SaaS shardmap for {}@{}. Child of {}".format(
                    saas_service.name, saas_service.ctype, self.id), service=saas_service.name,
                    ctype=saas_service.ctype, check_shardmaps=True, **dict(self.Parameters)).enqueue()
                subtasks.append(subtask)
            raise sdk2.WaitTask(subtasks, ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=True)
