import itertools
import logging
import requests

from datetime import date, datetime

import sandbox.sdk2 as sdk2

from saas.library.python.deploy_manager_api import SaasService
from saas.library.python.lunapark import lunapark_job
from saas.library.python.yasm import common
from saas.library.python.yasm.saas_service_metrics import SaasServiceMetrics

from sandbox.projects.saas.common.classes import DmProxyHandles

from startrek_client import Startrek
from startrek_client.exceptions import NotFound

from yandex_tracker_client.exceptions import UnprocessableEntity
from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient


KEY_METRICS = [
    'proxy_rps', 'proxy_cpu_avg', 'proxy_cpu_quant99', 'proxy_ram_avg', 'proxy_ram_quant99',
    'backend_cpu_avg', 'backend_cpu_quant99', 'backend_ram_avg', 'backend_ram_quant99'
]

LINEAR_METRICS = [
    'proxy_cpu_avg', 'proxy_cpu_quant99', 'proxy_ram_avg', 'proxy_ram_quant99', 'backend_cpu_avg',
    'backend_cpu_quant99', 'backend_ram_avg', 'backend_ram_quant99'
]

NORMALIZE_METRICS = ['proxy_rps', 'backend_rps']


def under_cut(text):
    return '<{' + text + '}>'


def link_format(href, text, description='', **kwargs):
    return '{}<a href=\'{}\'>{}</a>'.format(description, href, text)


def service_required(func):
    def wrapped(self, service=None, ctype=None, *args, **kwargs):
        if service and ctype:
            return func(self, *args, service=service, ctype=ctype, **kwargs)
        elif self.service and self.ctype:
            return func(self, *args, service=self.service, ctype=self.ctype, **kwargs)
        else:
            raise ValueError('Not stated ctype or service')
    return wrapped


class SingleShootingResult:

    YT_CLIENT = None
    YT_BASE_PATH = '//home/saas/shooting'

    @classmethod
    def init_yt_client(cls, yt_token, proxy='hahn'):
        cls.YT_CLIENT = YtClient(token=yt_token, proxy=proxy)

    @classmethod
    def fromdict(cls, data_dict):
        return cls(**data_dict)

    def __init__(self, geo, rps, task_id=None, service=None, ctype=None, target=None, yt_token=None):
        self.geo = geo
        self.rps = rps
        self.task_id = task_id
        self.target = target
        self.service = service
        self.ctype = ctype
        if not ShootingResults.YT_CLIENT:
            SingleShootingResult.init_yt_client(yt_token)

    @property
    def sandbox_task(self):
        return sdk2.Task[self.task_id]

    @property
    def lunapark_link(self):
        return self.sandbox_task.Parameters.lunapark_link

    @property
    def lunapark_job(self):
        return lunapark_job.LunaparkJob(self.sandbox_task.Parameters.lunapark_job_id)

    @property
    def job_time_range(self):
        return self.lunapark_job.time_range

    @service_required
    def task_links(self, service=None, ctype=None):
        return self.lunapark_job.golovan_links_from_shoot(service=service, ctype=ctype, geo=self.geo)

    @staticmethod
    def get_safe_yasm_snapshot(*args, **kwargs):
        import logging

        try:
            return common.get_golovan_snapshot(*args, **kwargs)
        except Exception as e:
            logging.exception(e)
            return 'Can\'t get snapshot'

    @service_required
    def ticket_info(self, service=None, ctype=None):
        info = {}
        task_links = self.task_links(service, ctype)

        info['common_result_info'] = 'dc: {}, rps: {}, target_rps: {}, lunapark_link: {}'.format(
            self.geo, self.rps, self.target, self.lunapark_link
        )
        info['lunapark_link_info'] = under_cut('lunapark link \n{}'.format(self.lunapark_link))
        info['metrics_links_info'] = '\n'.join([
            under_cut('{} \n{}'.format(name, self.get_safe_yasm_snapshot(task_links[name])))
            for name in task_links
        ])
        return info

    @service_required
    def store_metrics(self, task_id, service=None, ctype=None):
        schema = [{'name': 'ts', 'type': 'timestamp'}]
        for key in KEY_METRICS:
            schema.append({'name': key, 'type': 'double'})
        for key in LINEAR_METRICS:
            schema.append({'name': 'linear_' + key, 'type': 'double'})

        results = {}
        linear_results = {}

        metrics = SaasServiceMetrics(ctype, service)
        interval_begin, interval_end = self.job_time_range
        proxy_rps_list = list(metrics.proxy_rps(geo=self.geo)(interval_begin, interval_end, normalize=True))
        results['ts'], results['proxy_rps'] = zip(*proxy_rps_list)

        for column in KEY_METRICS:
            results[column] = getattr(metrics, column)(geo=self.geo).values(
                interval_begin, interval_end, normalize=True if column in NORMALIZE_METRICS else False
            )

        for column in LINEAR_METRICS:
            data = getattr(metrics, column)(geo=self.geo) if results.get(column) is None else results[column]
            results['linear_' + column], linear_results[column] = metrics.linear(
                results['proxy_rps'], data, interval_begin, interval_end, with_coefs=True
            )

        base_path = '{base}/{ctype}/{service}/{key}/metrics'.format(
            base=self.YT_BASE_PATH, ctype=ctype, service=service, key=task_id
        )

        dest_table = self.obtain_yt_table('{base}/{geo}'.format(base=base_path, geo=self.geo), schema)
        table = [{key: results[key][i] for key in results} for i in range(len(results['ts']))]
        self.YT_CLIENT.write_table(dest_table, table)

        schema = [
            {'name': 'metric', 'type': 'string'},
            {'name': 'coef', 'type': 'float'},
            {'name': 'bias', 'type': 'float'}
        ]
        dest_table = self.obtain_yt_table('{base}/linear_data_{geo}'.format(base=base_path, geo=self.geo), schema)
        table = [
            {
                'metric': column,
                'coef': linear_results[column][0],
                'bias': linear_results[column][1]
            } for column in LINEAR_METRICS
        ]
        self.YT_CLIENT.write_table(dest_table, table)

    @classmethod
    def obtain_yt_table(cls, path, schema=None):
        directory, dest_table = '/'.join(path.split('/')[:-1]), path
        if not cls.YT_CLIENT.exists(directory):
            cls.YT_CLIENT.mkdir(directory, recursive=True)
        if not cls.YT_CLIENT.exists(dest_table):
            cls.YT_CLIENT.create('table', dest_table, attributes={'schema': schema})
        return path

    def sandbox_info(self):
        info = {}

        info['common_result_info'] = 'dc: {}, rps: {}, target_rps: {}, lunapark_link: {}'.format(
            self.geo, self.rps, self.target, self.lunapark_link
        )
        info['lunapark_link_info'] = under_cut('lunapark link \n{}'.format(self.lunapark_link))

        return info

    def warden_repr(self):
        return {'rps': self.rps, 'dc': self.geo, 'target': self.target}

    def asdict(self):
        return {
            'geo': self.geo,
            'rps': self.rps,
            'task_id': self.task_id,
            'target': self.target,
            'service': self.service,
            'ctype': self.ctype
        }


class ShootingResults:
    ST_CLIENT = None
    WARDEN_CLIENT = None
    YT_CLIENT = None

    SETRACE_BASE_URL = 'https://setrace.yandex-team.ru/web/trace'
    YT_BASE_PATH = '//home/saas/shooting'

    @classmethod
    def init_yql_client(cls, yql_token):
        cls.YQL_CLIENT = YqlClient(token=yql_token)

    @classmethod
    def init_yt_client(cls, yt_token, proxy='hahn'):
        SingleShootingResult.init_yt_client(yt_token, proxy)
        cls.YT_CLIENT = YtClient(token=yt_token, proxy=proxy)

    @classmethod
    def init_startrek_client(cls, token):
        cls.ST_CLIENT = Startrek(api_version='v2', useragent='saas-robot', token=token)

    @classmethod
    def init_warden_client(cls, token):
        cls.WARDEN_CLIENT = requests.Session()
        cls.WARDEN_CLIENT.headers = {'Authorization': 'OAuth {}'.format(token)}

    @classmethod
    def fromdict(cls, data_dict):
        task_id = data_dict['task_id']
        service = data_dict['service']
        ctype = data_dict['ctype']
        results = data_dict['results']

        result_objects = []
        for result in results:
            result_objects.append(SingleShootingResult.fromdict(result))

        return cls(task_id=task_id, service=service, ctype=ctype, results=result_objects)

    def __init__(self, task_id, service, ctype, results=None, startrek_token=None, warden_token=None, yt_token=None):
        self.task_id = task_id
        self.service = service
        self.ctype = ctype
        self.results = results if results else []

        if not ShootingResults.YT_CLIENT:
            ShootingResults.init_yt_client(yt_token)
        if not ShootingResults.ST_CLIENT:
            ShootingResults.init_startrek_client(startrek_token)
        if not ShootingResults.WARDEN_CLIENT:
            ShootingResults.init_warden_client(warden_token)

    @property
    def component_name(self):
        return '{}_{}'.format(self.ctype.replace('-', '_'), self.service.replace('-', '_'))

    @property
    def full_component_name(self):
        return 'saas/{}'.format(self.component_name)

    def get_or_create_component(self):
        params = {'name': self.component_name, 'parent_component_name': 'saas'}
        resp = self.WARDEN_CLIENT.get(
            'http://warden.z.yandex-team.ru/api/warden.Warden/getComponent',
            json=params
        ).json()

        if not resp:
            component = {'name': self.component_name,
                         'parent_component': 'saas',
                         'owner_list': [{'login': login} for login in ['i024', 'salmin', 'vbushev']],
                         'spiChat': {'chatLink': 'https://t.me/joinchat/EwesklUsMAK9-GlsLYSRYg',
                                     'name': 'SpiCoordinationChat'}, 'tags': [{'tag': 'mass_shooting'}]}
            params = {'component': component, 'parent_component': 'saas'}
            self.WARDEN_CLIENT.post(
                'http://warden.z.yandex-team.ru/api/warden.Warden/createChildComponent',
                json=params
            )

        return self.full_component_name

    def store_metrics(self):
        for result in self.results:
            result.store_metrics(task_id=self.task_id, service=self.service, ctype=self.ctype)

    def _get_request_log_tables_path(self):
        return '//logs/saas/{ctype}/access-log/30min'.format(ctype=self.ctype)

    def _get_request_log_table_path(self, table_name):
        return '{base_path}/{table_name}'.format(base_path=self._get_request_log_tables_path(), table_name=table_name)

    @staticmethod
    def _get_table_ts(table_name):
        epoch = datetime(1970, 1, 1)
        msk_offset = 3 * 60 * 60  # UTC+3

        dtm = datetime.strptime('{name}'.format(name=table_name), '%Y-%m-%dT%H:%M:%S')
        start_ts = int((dtm - epoch).total_seconds()) - msk_offset
        end_ts = start_ts + 30 * 60

        return start_ts, end_ts

    def _get_relevant_request_log_tables(self, shooting_start_ts, shooting_end_ts):
        table_names = self.YT_CLIENT.list(self._get_request_log_tables_path())
        if not table_names:
            return None

        result = []
        table_names = list(reversed(sorted(table_names)))
        _, end_ts = self._get_table_ts(table_names[0])
        if end_ts < shooting_end_ts:
            return None

        for table_name in table_names:
            start_ts, end_ts = self._get_table_ts(table_name)

            if shooting_start_ts < end_ts and start_ts < shooting_end_ts:
                result.append(table_name)
            elif shooting_start_ts >= end_ts:
                break

        logging.info('Searchable tables for ts pair (%d, %d): %s', shooting_start_ts, shooting_end_ts, result)

        if not result:
            raise RuntimeError('Unable to find relevant tables for the given input data')

        return result

    def store_requests_info(
            self,
            find_by,
            shooting_start_ts,
            shooting_end_ts,
            startrek_ticket_id,
            proxy='hahn',
            ticket_reqids_cnt=10,
    ):
        relevant_tables = self._get_relevant_request_log_tables(shooting_start_ts, shooting_end_ts)
        if not relevant_tables:
            return False

        yql_results = []
        for table_name in relevant_tables:
            yql_query = """
SELECT *
FROM {proxy}.`{table_path}`
WHERE
`service`='{service}' and
`processed-query` LIKE '%{find_by}%' and
`http_status` != '200' and
`timestamp` >= {start_ts} and
`timestamp` < {end_ts}
            """.format(
                proxy=proxy,
                table_path=self._get_request_log_table_path(table_name),
                service=self.service,
                find_by=find_by,
                start_ts=shooting_start_ts,
                end_ts=shooting_end_ts,
            )

            request = self.YQL_CLIENT.query(yql_query, syntax_version=1)
            request.run()

            result = request.get_results()
            if not result.is_success:
                error_msg = '\n'.join([str(err) for err in result.errors])
                logging.error('Unable to execute YQL query: %s, errors: %s', yql_query, error_msg)
                raise RuntimeError('YQL query has failed!')

            yql_results = itertools.chain(yql_results, result.full_dataframe.iterrows())

        sample_table_path = self._get_request_log_table_path(table_name=relevant_tables[-1])
        dest_table_schema = self.YT_CLIENT.get('{table_path}/@schema'.format(table_path=sample_table_path))

        dest_table = SingleShootingResult.obtain_yt_table(
            '{base}/{ctype}/{service}/{key}/requests_info'.format(
                base=self.YT_BASE_PATH, ctype=self.ctype, service=self.service, key=self.task_id
            ),
            schema=[row for row in dest_table_schema if not row['name'].startswith('_')]
        )
        dest_data = [{k: v for k, v in row.items() if not k.startswith('_')} for _, row in yql_results]
        self.YT_CLIENT.write_table(dest_table, dest_data)

        reqids = []
        for idx, data in enumerate(dest_data):
            if idx >= ticket_reqids_cnt:
                break
            reqids.append(data['reqid'])

        text = 'The shooting is over!'
        if reqids:
            reqids_text = '\n'.join([
                '(({setrace}/{reqid} {reqid}))'.format(setrace=self.SETRACE_BASE_URL, reqid=reqid)
                for reqid in reqids
            ])
            cut = under_cut('Failed reqids [first {n}] \n{reqids}'.format(n=ticket_reqids_cnt, reqids=reqids_text))
            text = '{text}\n{cut}'.format(text=text, cut=cut)
        else:
            text = '{text}\nThere are no failed requests'.format(text=text)

        ticket = self.ST_CLIENT.issues[startrek_ticket_id]
        ticket.comments.create(text=text)
        return True

    def reduce_information_for_startrek(self):
        data = [{'geo': el.geo, 'value': el.ticket_info(service=self.service, ctype=self.ctype)} for el in self.results]
        comment = 'Metrics of shooting to saas service {} in ctype {}\n'.format(self.service, self.ctype)
        subject = 'Results of shooting to saas service {} in ctype {}\n'.format(self.service, self.ctype)
        head = 'Shoot to saas service {} in ctype {}'.format(self.service, self.ctype)
        for measurement in data:
            comment += under_cut('{} \n{}'.format(measurement['geo'], measurement['value']['metrics_links_info']))
            subject += measurement['value']['common_result_info'] + '\n'

        return {'comment': comment, 'subject': subject, 'head': head}

    def reduce_information_for_sandbox(self):
        result = ''
        for measurement in self.results:
            info = measurement.ticket_info(service=self.service, ctype=self.ctype)['common_result_info']
            result += info + '\n'
        return result

    def generate_ticket(self, main_ticket, responsibles=None):
        data_to_fill = self.reduce_information_for_startrek()

        report_issue = self.ST_CLIENT.issues.create(
            queue='SAASLOADTEST', summary=data_to_fill['head'], description=data_to_fill['subject'],
            assignee='bvdvlg', tags=['mass_shooting']
        )
        report_issue.comments.create(text=data_to_fill['comment'], summonees=responsibles)
        current_issue = self.ST_CLIENT.issues[report_issue['key']]

        try:
            current_issue.links.create(issue=main_ticket, relationship='relates')
        except UnprocessableEntity:
            logging.info('Link between {} and {} exists'.format(main_ticket, report_issue['key']))
        except NotFound:
            logging.info('Issue {} wasn\'t found'.format(main_ticket))

        report_issue.transitions['close'].execute(resolution='fixed')
        return report_issue['key']

    def send_to_uchenki(self, ticket):
        dm_proxy = DmProxyHandles()

        saas_service = SaasService(self.ctype, self.service)
        basic_locations_num = len(set(saas_service.locations).intersection({'MAN', 'SAS', 'VLA'}))

        res_by_dc = [
            {
                'dc': el.geo.upper(),
                'rps': el.rps,
                'target': int(dm_proxy.get_sla_rps(self.ctype, self.service)) / basic_locations_num
            } for el in self.results
        ]
        uchenki_result = {
            'issue': ticket,
            'full_component_name': self.full_component_name,
            'date': str(date.today()),
            'result': res_by_dc
        }

        self.WARDEN_CLIENT.post(
            'https://uchenki.yandex-team.ru/api/v1/training/training_results',
            json=uchenki_result
        )

    def asdict(self):
        return {
            'results': [el.asdict() for el in self.results],
            'service': self.service,
            'ctype': self.ctype,
            'task_id': self.task_id
        }

    def __getitem__(self, item):
        for res in self.results:
            if res.geo == item:
                return res
