import collections
import json
import logging
import requests

import sandbox.common.types.task as ctt

from sandbox import sdk2


KILL_TIMEOUT = 6 * 60 * 60
SANDBOX_OWNER = 'ANTISPAM-ROBOT'
TEST_REQUESTS = 1347882165


class ResponseStatistics:
    MIN_REQUEST_COUNT = 1
    MIN_RESPONSE_RATIO = 0.5
    MAX_HTTP_ERROR_COUNT = 0
    MAX_ERROR_COUNT = 0

    def __init__(self, service='', request_count=0, response_count=0, retry_count=0, http_error_count=0, error_count=0, timeout_count=0, http_error_messages={}, error_messages={}):
        self.service = service
        self.request_count = request_count
        self.response_count = response_count
        self.retry_count = retry_count
        self.http_error_count = http_error_count
        self.error_count = error_count
        self.timeout_count = timeout_count
        self.http_error_messages = collections.Counter(http_error_messages)
        self.error_messages = collections.Counter(error_messages)

    def to_json(self):
        return json.dumps(self.__dict__)

    @classmethod
    def from_json(cls, payload):
        return cls(**json.loads(payload))

    def register_request(self, retry):
        if retry == 0:
            self.request_count += 1
        else:
            self.retry_count += 1

    def register_response(self, response):
        need_retry = False

        if response.status_code != requests.codes.ok:
            self.http_error_count += 1
            self.http_error_messages['{} {}'.format(response.status_code, response.reason)] += 1
            need_retry = True
            return need_retry

        result = json.loads(response.text)['result']
        errors = result.get('errors', {})

        if 'clean-web' in errors:
            if errors['clean-web'] == 'timeout':
                self.timeout_count += 1
                need_retry = True
            else:
                self.error_count += 1
                self.error_messages[errors['clean-web']] += 1
        else:
            self.response_count += 1

        return need_retry

    def error(self):
        if self.request_count < self.MIN_REQUEST_COUNT:
            return 'Too few requests in resource #{} for service {}: {}, minimum is {}'.format(
                TEST_REQUESTS,
                self.service,
                self.request_count,
                self.MIN_REQUEST_COUNT,
            )

        if float(self.response_count) / self.request_count < self.MIN_RESPONSE_RATIO:
            return 'Too low response/request ratio for service {}: {}/{} = {:.3f}, minimum is {}'.format(
                self.service,
                self.response_count,
                self.request_count,
                float(self.response_count) / self.request_count,
                self.MIN_RESPONSE_RATIO,
            )

        if self.http_error_count < self.MAX_HTTP_ERROR_COUNT:
            return 'Too many HTTP errors for service {}: {}, maximum is {}; messages: {}'.format(
                self.service,
                self.http_error_count,
                self.MAX_HTTP_ERROR_COUNT,
                self.http_error_messages,
            )

        if self.error_count < self.MAX_ERROR_COUNT:
            return 'Too many errors for service {}: {}, maximum is {}; messages: {}'.format(
                self.service,
                self.error_count,
                self.MAX_ERROR_COUNT,
                self.error_messages,
            )

        return ''


class CleanWebSmokeTest(sdk2.Task):
    ROUTER = 'http://cw-router-dev.common.yandex.net/v2/'
    CONNECT_TIMEOUT = 10.0
    READ_TIMEOUT = 10.0
    MAX_RETRIES = 10

    class Context(sdk2.Context):
        statistics = None

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = KILL_TIMEOUT

        service_list = sdk2.parameters.String('Select services (semicolon separated)', required=True, default_value='')

        owner = sdk2.parameters.String('Owner', required=True, default_value=SANDBOX_OWNER)
        description = 'Smoke test for Clean Web'
        priority = ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.NORMAL)

    def _serialise_statistics(self, statistics):
        self.Context.statistics = json.dumps([
            service_statistics.to_json()
            for service_statistics in statistics.values()
        ])

    def _deserialise_statistics(self):
        return {
            service_statistics.service: service_statistics
            for service_statistics in map(lambda x: ResponseStatistics().from_json(x), json.loads(self.Context.statistics))
        }

    def _make_request(self, request, request_id=0, timeout=None):
        rpc_request = {
            'jsonrpc': '2.0',
            'method': 'process',
            'params': request,
            'id': request_id,
        }

        headers = {'Content-Type': 'application/json'}
        data = json.dumps(rpc_request)

        response = requests.post(self.ROUTER, data=data, headers=headers, timeout=timeout)

        if response.status_code != 200:
            logging.error('Got HTTP error "{} {}" from {} for POST of data "{}" with headers "{}"'.format(response.status_code, response.reason, self.ROUTER, data, headers))

        return response

    def _test_requests(self, services):
        statistics = {
            service: ResponseStatistics(service)
            for service in services
        }

        test_requests = sdk2.ResourceData(sdk2.Resource[TEST_REQUESTS])

        with test_requests.path.open('r', encoding='utf8') as test_requests_file:
            for request_id, line in enumerate(test_requests_file.readlines()):
                line = line.strip()
                request = json.loads(line)

                if 'request' in request:
                    request = request['request']

                if request['service'] in services:
                    proceed = True
                    retry = 0

                    while proceed and retry < self.MAX_RETRIES:
                        statistics[request['service']].register_request(retry)

                        response = self._make_request(
                            request,
                            request_id=request_id,
                            timeout=(self.CONNECT_TIMEOUT, self.READ_TIMEOUT),
                        )

                        proceed = statistics[request['service']].register_response(response)

                        retry += 1

        self._serialise_statistics(statistics)

        broken_services = list(filter(lambda service: statistics[service].error(), services))

        if broken_services:
            raise Exception(
                'Test failed, these services are broken: {}.'.format(', '.join(broken_services))
            )

    @sdk2.header()
    def test_results(self):
        if self.Context.statistics is None:
            return 'No test results yet.'

        results = self._deserialise_statistics()

        report = '<h3>Test results</h3>'
        report += '<style>th, td { padding: 5px; }</style>'
        report += '<table><tr><th>Service</th><th>Result</th><th>Responses</th><th>Requests</th><th>Retries</th><th>HTTP errors</th><th>Service errors</th><th>Error message</th></tr>'

        for service in sorted(results.keys()):
            service_result = results[service]
            error_message = service_result.error()
            report += '<tr><td>{}</td><td>{}</td><td>{}</td><td>{}</td><td>{}</td><td>{}</td><td>{}</td><td>{}</td></tr>'.format(
                service,
                '<font color="red">Failed</font>' if error_message else '<font color="green">Ok</font>',
                service_result.response_count,
                service_result.request_count,
                service_result.retry_count,
                service_result.http_error_count,
                service_result.error_count,
                error_message,
            )

        report += '</table>'

        return report

    def on_prepare(self):
        self._services = set(self.Parameters.service_list.split(';'))

    def on_execute(self):
        self._test_requests(self._services)
