import logging

import sandbox.common.types.task as ctt
import sandbox.projects.common.build.YaPackage as YaPackage
import sandbox.projects.common.build.parameters as build_params

from sandbox import sdk2
from sandbox.projects.common.build.arcadia_project_misc import get_arcadia_project_base_target_params
from sandbox.projects.mail.CommonLib.lib.docker import get_image_hash
from sandbox.projects.mail.CommonLib.lib.qloud import QloudApi
from sandbox.projects.safesearch.CleanWebTestAndDeploy.config.config import Config, PROJECTS
from sandbox.sandboxsdk.svn import Arcadia


JOB_FAIL = 'Fail'
JOB_SUCCESS = 'Success'
KILL_TIMEOUT = 6 * 60 * 60

STAGE_DOCKER = '1'
STAGE_DEV = '2'
STAGE_TEST_DEV = '3'
STAGE_PROD = '4'


class CleanWebTestAndDeploy(sdk2.Task):
    class Context(sdk2.Context):
        arcadia_url = None
        build_revision = None

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = KILL_TIMEOUT

        with sdk2.parameters.CheckGroup('Select projects') as projects_multiple:
            for project in PROJECTS:
                no_tests = '[no tests available] ' if project.service is None else ''
                caption = '{}{}\t({})'.format(no_tests, project.title, ','.join(map(lambda c: c.path, project.components)))
                projects_multiple.values[project.title] = projects_multiple.Value(caption, checked=False)

        owner = sdk2.parameters.String('Owner', required=True, default_value=Config.SANDBOX_OWNER)
        description = 'Create Clean Web Release: build and push docker image, deploy to dev, test in dev, deploy to prod'
        priority = ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.NORMAL)

        arcadia_url = sdk2.parameters.ArcadiaUrl('Svn url (you can add "@<commit_number>" or specify branch path)', default_value=build_params.ArcadiaUrl.default_value)
        review_id = sdk2.parameters.Integer('Arcanum review number, will apply patch onto arcadia url (optional)', default_value=None)

        qloud_token_vault_owner = sdk2.parameters.String('Vault owner for Qloud OAuth token', default_value='')
        qloud_token_vault_item = sdk2.parameters.String('Vault item containing Qloud OAuth token', required=True, default_value=Config.QLOUD_TOKEN_VAULT_NAME)

        docker_user_name = sdk2.parameters.String('Docker user name', required=True, default_value=Config.DOCKER_USER_NAME)
        docker_token_vault_item = sdk2.parameters.String('Vault item containing Docker OAuth token', required=True, default_value=Config.DOCKER_TOKEN_VAULT_NAME)

        with sdk2.parameters.RadioGroup('Deploy stage') as stage:
            stage.values[STAGE_DOCKER] = stage.Value('Build docker images')
            stage.values[STAGE_DEV] = stage.Value('... and deploy them to dev')
            stage.values[STAGE_TEST_DEV] = stage.Value('... and run smoke tests for them in dev')
            stage.values[STAGE_PROD] = stage.Value('... and deploy them to prod', default=True)

    def _ya_make_test_subtask(self, projects):
        task_class = sdk2.Task['YA_MAKE']

        tests = set(Config.CLEAN_WEB_TESTS)
        for project in projects:
            for component in project.components:
                project_path = '/'.join(component.path.split('/')[:-1])
                tests.add('{}/{}'.format(Config.CLEAN_WEB_ROOT, project_path))

        kwargs = {
            build_params.ArcadiaUrl.name: self.__arcadia_url,
            build_params.ArcadiaPatch.name: self.__arcadia_patch,
            build_params.BuildType.name: 'release',
            build_params.UseArcadiaApiFuse.name: True,
            get_arcadia_project_base_target_params().TargetsParameter.name: ';'.join(tests),
            build_params.TestParameter.name: True,
        }

        kwargs.update(self.__common_subtask_args)

        sub_task = task_class(
            self,
            description='Subtask for Clean Web tests',
            **kwargs
        ).enqueue()

        return sub_task

    def _ya_package_docker_bulk(self, projects):
        task_class = sdk2.Task['YA_PACKAGE']

        all_packages = set()
        for project in projects:
            for component in project.components:
                all_packages.add('{}/{}/package.json'.format(Config.CLEAN_WEB_ROOT, component.path))

        kwargs = {
            build_params.ArcadiaUrl.name: self.__arcadia_url,
            build_params.ArcadiaPatch.name: self.__arcadia_patch,
            build_params.BuildType.name: 'release',
            build_params.UseArcadiaApiFuse.name: True,
            YaPackage.PackagesParameter.name: ';'.join(all_packages),
            YaPackage.PackageTypeParameter.name: YaPackage.DOCKER,
            YaPackage.DockerPushImageParameter.name: True,
            YaPackage.DockerRegistryParameter.name: 'registry.yandex.net',
            YaPackage.DockerUserParameter.name: self.Parameters.docker_user_name,
            YaPackage.DockerTokenVaultName.name: self.Parameters.docker_token_vault_item,
            YaPackage.DockerImageRepositoryParameter.name: Config.DOCKER_REPOSITORY,
            YaPackage.CustomVersionParameter.name: self.__custom_version,
            'ya_yt_store': True,
            'ya_yt_token_vault_owner': Config.YT_TOKEN_VAULT_OWNER,
            'ya_yt_token_vault_name': Config.YT_TOKEN_VAULT_NAME,
        }
        kwargs.update(self.__common_subtask_args)
        sub_task = task_class(
            self,
            description='Subtask for Clean Web docker build',
            **kwargs
        ).enqueue()
        return sub_task

    def _update_qloud_environment_bulk_subtask(self, projects, environment_type='dev'):
        def get_environment_id(project):
            return '.'.join([Config.ENVIRONMENT_PREFIX, project.environment_id, environment_type])

        token = sdk2.Vault.data(self.Parameters.qloud_token_vault_owner or self.author, self.Parameters.qloud_token_vault_item)
        api = QloudApi(url='https://platform.yandex-team.ru', token=token, logger=logging.getLogger('QloudApi'), deploy_timeout=3*3600)

        for project in projects:
            params = []
            for component in project.components:
                image_url = 'registry.yandex.net/{}/{}:{}'.format(Config.DOCKER_REPOSITORY, component.path.replace('clients/', '').replace('/', '_'), self.__custom_version)
                image_hash = get_image_hash(*image_url.split(':'))

                params.append({
                    'componentName': component.component_name,
                    'properties': {'repository': image_url, 'hash': image_hash}
                })
            logging.info('Parameters to deploy: ' + str(params))

            environment_id = get_environment_id(project)

            version = api.get_environment_deployed_version(environment_id)
            if not version:
                error_message = 'Can not get deployed version of environment_id={}, check project status is "DEPLOYED"'.format(environment_id)
                logging.error(error_message)
                raise Exception(error_message)

            api.fast_deploy_environment(environment_id=environment_id,
                                        version=version,
                                        params=params)

        map(api.wait_for_environment_deploy, map(get_environment_id, projects))

    def _smoke_test_subtask(self, projects):
        task_class = sdk2.Task['CLEAN_WEB_SMOKE_TEST']

        service_list = []
        for project in projects:
            if project.service is not None:
                service_list.append(project.service)

        kwargs = {
            'service_list': ';'.join(service_list),
        }

        kwargs.update(self.__common_subtask_args)

        sub_task = task_class(
            self,
            description='Subtask for Clean Web smoke tests',
            **kwargs
        ).enqueue()

        return sub_task

    def _get_job_status(self):
        if not all(map(lambda sub_task: sub_task.status == ctt.Status.SUCCESS, self.find(children=True))):
            return JOB_FAIL
        else:
            return JOB_SUCCESS

    def on_save(self):
        parsed_url = Arcadia.parse_url(self.Parameters.arcadia_url)
        if not parsed_url.revision:
            self.Context.build_revision = Arcadia.get_revision(self.Parameters.arcadia_url)
            self.Context.arcadia_url = self.Parameters.arcadia_url + '@' + str(self.Context.build_revision)
        else:
            self.Context.build_revision = parsed_url.revision
            self.Context.arcadia_url = self.Parameters.arcadia_url

    def on_prepare(self):
        logging.info('Revision for build docker image and ya make subtasks: ' + str(self.Context.build_revision))

        self.__arcadia_url = self.Context.arcadia_url
        self.__arcadia_patch = 'arc:{}'.format(self.Parameters.review_id) if self.Parameters.review_id else None
        if self.Parameters.review_id:
            self.__custom_version = 'r{}-a{}'.format(self.Context.build_revision, self.Parameters.review_id)
        else:
            self.__custom_version = 'r{}'.format(self.Context.build_revision)

        self.__common_subtask_args = {
            'owner': self.Parameters.owner,
            'priority': self.Parameters.priority,
            'notifications': self.Parameters.notifications,
            'kill_timeout': KILL_TIMEOUT,
        }

    def on_execute(self):
        projects = list(filter(lambda x: x.title in self.Parameters.projects_multiple, PROJECTS))

        with self.memoize_stage.ya_make_tests:
            ya_make_test_sub_task = self._ya_make_test_subtask(projects)
            sub_tasks = [ya_make_test_sub_task.id]
            raise sdk2.WaitTask(sub_tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage.run_docker_image_build:
            if self._get_job_status() == JOB_SUCCESS:
                ya_package_docker_sub_task = self._ya_package_docker_bulk(projects)
                sub_tasks = [ya_package_docker_sub_task.id]
                raise sdk2.WaitTask(sub_tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage.deploy_project_to_dev:
            if self._get_job_status() == JOB_SUCCESS and self.Parameters.stage >= STAGE_DEV:
                self._update_qloud_environment_bulk_subtask(projects, 'dev')

        with self.memoize_stage.test_dev:
            if self._get_job_status() == JOB_SUCCESS and self.Parameters.stage >= STAGE_TEST_DEV:
                smoke_test_sub_task = self._smoke_test_subtask(projects)
                sub_tasks = [smoke_test_sub_task.id]
                raise sdk2.WaitTask(sub_tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage.deploy_project_to_prod:
            if self._get_job_status() == JOB_SUCCESS and self.Parameters.stage >= STAGE_PROD:
                self._update_qloud_environment_bulk_subtask(projects, 'prod')

        with self.memoize_stage.finish_job:
            if self._get_job_status() == JOB_FAIL:
                failed_sub_tasks = filter(lambda sub_task: sub_task.status != ctt.Status.SUCCESS, self.find(children=True))
                failed_tasks_ids_types = map(lambda sub_task: [sub_task.id, sub_task.type], failed_sub_tasks)
                raise Exception('Following subtasks failed: ' + str(failed_tasks_ids_types))
