import datetime
import gevent
import os

from sandbox import sdk2
from sandbox.sandboxsdk import environments


class TolokaAnswerChecker(object):
    def __init__(self):
        self.error_was_found = False
        self.checkers = {
            'common_toloka_results': self.check_common_toloka_invariants,
        }

    def check_common_toloka_invariants(self, raw_result):
        for verdict, expected_degrees in (
            ('porno', ['light_vulgarity', 'porno']),
            ('swearing', ['rude', 'light_obscene', 'hard_obscene']),
            ('insult', ['light', 'hard']),
            ('threat', ['light', 'hard']),
        ):
            degree = 'degree_' + verdict

            if raw_result.get(verdict) is True:
                if degree not in raw_result:
                    return '{!r} is True, but {!r} is missing.'.format(verdict, degree)

                if raw_result.get(degree) not in expected_degrees:
                    return 'Unknown {!r} value {!r}. Expected values: {!r}.'.format(
                        degree,
                        raw_result.get(degree),
                        expected_degrees,
                    )

            if degree in raw_result:
                if raw_result.get(verdict) is not True:
                    return '{!r} is present, but {!r} = {!r} (expected to be True).'.format(
                        degree,
                        verdict,
                        raw_result.get(verdict)
                    )

        return ''

    def __call__(self, row):
        if self.error_was_found:
            return

        if row['event_name'] != 'toloka-end':
            return

        event_value = row['event_value']

        toloka_name = event_value['unified_result'].keys()[0]

        if toloka_name in self.checkers.keys():
            for raw_result in event_value['raw_results']:
                check_result = self.checkers[toloka_name](raw_result)

                if check_result:
                    self.error_was_found = True
                    yield {'toloka_name': toloka_name, 'error_message': check_result}
                    return


class CleanWebTolokaInvariantsTest(sdk2.Task):
    """Tests for non-trivial toloka verdicts invariants.

    This means checking (for each informative verdict) the conditions required for
    this verdict to be taken into account.

    Verdicts are considered informative and consistent (as trivial invariants).

    """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment("yandex-yt"),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
        ]

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("YT"):
            yt_token_owner = sdk2.parameters.String('Vault owner for YT token')
            yt_token_name = sdk2.parameters.String('Vault item containing YT Token')

    def on_execute(self):
        import yt.wrapper as yt

        toloka_answer_checker = TolokaAnswerChecker()

        logs_date = datetime.datetime.now() - datetime.timedelta(days=2)
        table_name = logs_date.strftime('%Y-%m-%d')

        input_table = '//home/logfeller/logs/clean-web-prod-events/1d/' + table_name
        output_table = '//home/antispam/clean_web/tests/toloka_invariants/' + table_name

        client = yt.YtClient(
            proxy='arnold',
            token=sdk2.Vault.data(self.Parameters.yt_token_owner, self.Parameters.yt_token_name),
            config={
                'pickling': {
                    'python_binary': '/skynet/python/bin/python',
                    # 'ImportError: cannot import name signal' fix - https://ml.yandex-team.ru/thread/yt/167477611142886638/
                    'additional_files_to_archive': [
                        (os.path.join(os.path.dirname(gevent.__file__), 'signal.py'), 'gevent/signal.py')
                    ]
                }
            }
        )

        if client.exists(output_table) is False:
            client.create('table', output_table)

        client.run_map(toloka_answer_checker, input_table, output_table)

        assert(client.row_count(output_table) == 0)
