from __future__ import unicode_literals

import logging
import json
import re
import requests
import six
from itertools import chain

from sandbox.projects.common.decorators import retries

logger = logging.getLogger(__name__)


class SamogonPkg(object):
    pkg_type = None

    def __init__(self, resource):
        self._resource = resource

    @property
    def payload(self):
        raise NotImplementedError

    def __str__(self):
        return "{}({})".format(self.__class__.__name__, self.payload)

    __repr__ = __str__


class SamogonPkgSandbox(SamogonPkg):
    pkg_type = "SANDBOX"

    @property
    def payload(self):
        return "sbr:{}".format(self._resource.id)


class SamogonPkgSkynet(SamogonPkg):
    pkg_type = "SKYNET"

    @property
    def payload(self):
        return str(self._resource.skynet_id)


SAMOGON_PKG = {
    SamogonPkgSandbox.pkg_type: SamogonPkgSandbox,
    SamogonPkgSkynet.pkg_type: SamogonPkgSkynet,
}


class SamogonBaseAPI(object):
    PACKAGE_PATTERN = re.compile(r"sbr:(\d+)")

    def __init__(self, base_url, token=None):
        self.base_url = base_url
        self._token = token

    @property
    def headers(self):
        if self._token is not None:
            return {
                "Authorization": "OAuth {}".format(self._token)
            }
        return {}

    @staticmethod
    def _subpath(*args):
        return "/".join(i.strip("/") for i in map(six.text_type, args))

    @retries(10, max_delay=10)
    def _do_request(self, path, method="GET", **kwargs):
        url = "{}/{}".format(self.base_url, path)
        r = requests.request(method.lower(), url, headers=self.headers, **kwargs)
        logger.debug("Response from %s: %s", url, r.content)
        r.raise_for_status()
        return r


class SamogonDeployAPI(SamogonBaseAPI):
    def deploy(self, pkg, project, namespace):
        path = self._subpath("deploy", project, namespace)
        self._do_request(path, method="POST", json={"pkg_id": pkg.payload})

    def upsert_secrets(self, secrets, project, namespace):
        path = self._subpath("upsert_secrets", project, namespace)
        self._do_request(path, method="POST", json=secrets)


class SamogonClusterAPI(SamogonBaseAPI):
    @classmethod
    def _extract_resource_ids_from_pkg(cls, pkg_str):
        resource_ids = cls.PACKAGE_PATTERN.findall(pkg_str)
        return resource_ids

    def get_meta_package(self):
        path = self._subpath("meta", "package")
        r = self._do_request(path)
        pkg_str = r.json()["value"]
        return pkg_str

    def get_meta_resource_id(self):
        pkg_str = self.get_meta_package()
        resource_id = int(self._extract_resource_ids_from_pkg(pkg_str)[0])
        return resource_id

    def get_deployed_packages(self):
        path = self._subpath("hosts", "versions")
        r = self._do_request(path, method="GET")
        info_by_hosts = r.json()["value"]
        return info_by_hosts

    def get_deployed_servant_resource_ids(self, environment, servant):
        path = self._subpath("environments", environment, servant)
        r = self._do_request(path, method="GET")
        info_by_hosts = r.json()["value"]
        return self._extract_resource_ids_host_info(info_by_hosts, path)

    def get_deployed_resource_ids(self):
        logger.info("Start getting resources for %s", self.base_url)
        hosts_infos = self.get_deployed_packages()
        return self._extract_resource_ids_host_info(hosts_infos, "/hosts/versions")

    @classmethod
    def _extract_resource_ids_host_info(cls, hosts_infos, url):
        resources = {}
        for host, info in hosts_infos.items():
            package = info["package"]
            try:
                if isinstance(package, list):
                    resource_ids = [cls._extract_resource_ids_from_pkg(x)[0] for x in package]
                elif isinstance(package, six.string_types):
                    resource_ids = cls._extract_resource_ids_from_pkg(package)
                else:
                    logger.error("%s handle returned unexpected data for %s: %s", url, host)
                    resource_ids = [0]
            except IndexError:
                logger.error(
                    "Something unexpected deployed at {url}::{host}: {pkg} is not a valid sandbox resource identifier".format(
                        url=url,
                        host=host,
                        pkg=package,
                    ))
                resource_ids = [0]
            resources[host] = resource_ids
        logger.info("Detected resources for %s:\n%s", url, json.dumps(resources, indent=2))
        return list(set(chain(*resources.values())))
