from __future__ import unicode_literals

import json
import logging
from copy import deepcopy
from sandbox.sdk2 import (
    Requirements,
    Resource,
    Vault,
    Task,
    parameters,
)
from sandbox.common.types import (
    resource,
    task,
)
from sandbox.projects.samogon import api as samogon_api


UNRELEASED = "UNRELEASED"


class DeploySamogonServants(Task):
    """Schedule Samogon deploy via UI API
    """
    class Requirements(Requirements):
        kill_timeout = 15 * 60  # 15 min
        cores = 1

        class Caches(Requirements.Caches):
            pass

    class Parameters(Task.Parameters):
        with parameters.Group("Project") as project_parameters:
            project = parameters.String("Project to deploy", required=True)
            namespace = parameters.Integer("Namespace to deploy", required=True, default_value=None)

        with parameters.Group("Package") as package_parameters:
            with parameters.String(
                "Resource storage to fetch package from",
                default=samogon_api.SamogonPkgSandbox.pkg_type
            ) as resource_storage:
                resource_storage.values.SANDBOX = samogon_api.SamogonPkgSandbox.pkg_type
                resource_storage.values.SKYNET = samogon_api.SamogonPkgSkynet.pkg_type

            auto = parameters.Bool("Automatically search for available resource", default=False)
            with auto.value[True]:
                resource_type = parameters.String("Resource type to search", default_value=None)
                with parameters.String("Release version", default=task.ReleaseStatus.STABLE) as release_version:
                    for release_type_name, release_type_value in task.ReleaseStatus.iteritems():
                        release_version.values[release_type_name] = release_type_value
                    release_version.values[UNRELEASED] = UNRELEASED

            with auto.value[False]:
                resource = parameters.Resource("Resource to deploy", description="Resource with samogon servants to deploy")

        with parameters.Group("Secrets") as vault_parameters:
            secrets = parameters.Dict("Secrets to use", description="Samogon secret key to Sandbox vault key mapping")

        with parameters.Group("Samogon parameters") as samogon_parameters:
            samogon_api_base_url = parameters.String(
                "Available samogon endpoints",
                default_value="https://api-deploy.n.yandex-team.ru"
            )
            samogon_oauth_token_vault_name = parameters.String("Samogon OAuth token Vault name", required=True)

    def find_new_package_id(self):
        if self.Parameters.auto:
            search_params = {
                "state": resource.State.READY,
                "order": "-id",
                "limit": 1,
            }
            logging.debug("Trying to find resource by:\n%s", json.dumps(search_params, indent=2))
            package_resource = self.find_resource(self.Parameters.resource_type, self.Parameters.release_version, search_params)
            if package_resource is None:
                raise Exception("Cannot find any '{}' resource released in '{}' or greater".format(self.Parameters.resource_type, self.Parameters.release_version))

            logging.info("Found %s resource", str(package_resource.id))
            return samogon_api.SAMOGON_PKG[self.Parameters.resource_storage](package_resource)
        else:
            return samogon_api.SAMOGON_PKG[self.Parameters.resource_storage](self.Parameters.resource)

    def find_resource(self, resource_type, released_ge, search_params):
        available_resources = []
        if released_ge != UNRELEASED:
            for release_type_name, release_type_value in task.ReleaseStatus.items():
                found_resource = self._find_exact_resource(resource_type, release_type_value, search_params)
                if found_resource is not None:
                    available_resources.append(found_resource)
                if release_type_name == released_ge:
                    break
        else:
            found_resource = self._find_exact_resource(resource_type, UNRELEASED, search_params)
            if found_resource is not None:
                available_resources.append(found_resource)

        if len(available_resources) > 0:
            package_resource = max(available_resources, key=lambda x: x.id)
            logging.info("Found %d resource", package_resource.id)
            return package_resource

    @staticmethod
    def _find_exact_resource(resource_type, released, search_params):
        _search_params = deepcopy(search_params)
        if released != UNRELEASED:
            _search_params.setdefault("attrs", {}).update({"released": released.lower()})
        return Resource[resource_type].find(**_search_params).first()

    def on_execute(self):
        token = Vault.data(self.Parameters.samogon_oauth_token_vault_name)
        deploy_api = samogon_api.SamogonDeployAPI(self.Parameters.samogon_api_base_url, token)

        secrets = {secret_name: Vault.data(vault_name) for secret_name, vault_name in self.Parameters.secrets.items()}
        if secrets:
            deploy_api.upsert_secrets(secrets, self.Parameters.project, self.Parameters.namespace)

        resource_to_deploy = self.find_new_package_id()
        deploy_api.deploy(resource_to_deploy, self.Parameters.project, self.Parameters.namespace)
