# -*- coding: utf-8 -*-

import email.mime.text
import logging
import re
import requests
import smtplib
import xml.etree.ElementTree
import json


from sandbox import sdk2

LOG_FORMAT = '%(asctime)-15s %(levelname)s [%(threadName)s] %(message)s'

JUGGLER_RULE_ID = '5b55d5d7a609120096a701c5'

CALENDAR_URL = "https://calendar.yandex-team.ru"
STAFF_API = "http://api.staff.yandex-team.ru/v3"
TELEGRAPH_API = "https://telegraph.yandex.net/api/v3"
JUGGLER_API = 'https://juggler-api.yandex-team.ru/api'

SAMOVAR_TEAM = {
        "marakasov": "9161365198",
        "dpsilaev": "9672595813",
        "kaikash7": "9168016997",
        "andshay": "9165591996",
        "zosimov": "9150509834",
        "alexcoach": "9168265782"
}


class Person:
    Login = None
    Phone = None

    @staticmethod
    def create(login, phone):
        person = Person()
        person.Login, person.Phone = login, phone
        return person


class ArrangeSamovarDuties(sdk2.Task):

    Robots = {
        "tier-1": ("robot-samovar-duty-1", "45420"),
        "tier-2": ("robot-samovar-duty-2", "45421"),
    }

    def get_robot_telegraph_token(self, robot_name):
        if robot_name == "robot-samovar-duty-1":
            return sdk2.Vault.data('SAMOVAR_ROBOT_DUTY1_TELEGRAPH_TOKEN')
        if robot_name == "robot-samovar-duty-2":
            return sdk2.Vault.data('SAMOVAR_ROBOT_DUTY2_TELEGRAPH_TOKEN')
        else:
            raise Exception("There is no telegraph token for robot %s" % robot_name)

    def get_current_juggler_rule(self, rule_id):
        url = JUGGLER_API + '/notify_rules/get_notify_rules?do=1'
        headers = {'Content-Type': 'application/json', 'Authorization': 'OAuth %s' % sdk2.Vault.data('SAMOVAR_ROBOT_JUGGLER_TOKEN')}
        data = {'filters': [{'rule_id': rule_id}]}
        response = requests.post(url, headers=headers, data=json.dumps(data))
        if not response.ok:
            raise Exception('Failed to get juggler rule %s.\nStatus code %s\nText: %s' % (
                                rule_id, response.status_code, repr(response.text)))

        response = json.loads(response.text)
        if response['total'] != 1:
            raise Exception('Failed to get juggler rule %s: no such rule' % (
                                rule_id))

        return response['rules'][0]

    def send_juggler_rule(self, rule):
        url = JUGGLER_API + '/notify_rules/add_or_update_notify_rule?do=1'
        headers = {'Content-Type': 'application/json', 'Authorization': 'OAuth %s' % sdk2.Vault.data('SAMOVAR_ROBOT_JUGGLER_TOKEN')}
        data = rule
        response = requests.post(url, headers=headers, data=json.dumps(data))
        if not response.ok:
            raise Exception('Failed to update juggler rule %s.\nStatus code %s\nText: %s' % (
                                rule['rule_id'], response.status_code, repr(response.text)))

    def update_juggler_rule(self, rule_id, duties):
        logging.info("Setting juggler rule %s notifications to %s and %s" % (rule_id, duties['tier-1'].Login, duties['tier-2'].Login))
        old_rule = self.get_current_juggler_rule(rule_id)
        rule = {}

        for f in ['rule_id', 'selector', 'template_name', 'template_kwargs', 'namespace', 'description', 'match_raw_events']:
            rule[f] = old_rule[f]

        rule['template_kwargs']['login'] = [duties['tier-1'].Login, duties['tier-2'].Login]
        self.send_juggler_rule(rule)

    def set_on_duty(self, robot, person):
        url = "%s/cucm/translation_pattern/pattern/%s" % (TELEGRAPH_API, robot[1])
        token = self.get_robot_telegraph_token(robot[0])
        headers = {'Authorization': 'OAuth %s' % token, 'Content-Type': 'application/json'}
        data = {'calledPartyTransformationMask': '98%s' % person.Phone}
        logging.info("Setting robot %s redirect to %s's phone" % (robot[0], person.Login))
        response = requests.put(url, headers=headers, json=data)
        if not response.ok:
            raise Exception("Failed to set redirect on %s.\nStatus code %s\nText: %s" % (
                                robot[0], response.status_code, repr(response.text)))

    def get_current_duties(self):
        url = "%s/export/rss.xml" % CALENDAR_URL
        calendar_privat_token = sdk2.Vault.data('SAMOVAR_DUTIES_CALENDAR_TOKEN')
        parametrs = {'private_token': calendar_privat_token, 'limit': "3"}
        response = requests.get(url, params=parametrs)
        logging.warning("code: %i, text:\n%s" % (response.status_code, response.text))
        if not response.ok:
            raise Exception("Failed to get calendar feed.\nStatus code: %s\nText: %s" % (
                                response.status_code, response.text))
        xml_parsed = xml.etree.ElementTree.fromstring(response.text.encode('utf-8'))
        entries = [e for e in xml_parsed if e.tag.endswith('entry')]

        def get_by_tag(parent, tagname):
            for e in parent:
                if e.tag.endswith(tagname):
                    return e
            return None

        duties = {}
        for entry in entries:
            node = get_by_tag(entry, 'title')
            if node is None:
                raise Exception("Entry with no title in calendar's feed")
            if re.match('tier\-1\: [a-zA-Z0-9\_\-]+@ tier\-2\: [a-zA-Z0-9\_\-]+@', node.text) is None:
                logging.warning("Bad samovar duties calendar entry. Title: %s" % node.text)
                continue
            tier1, name1, tier2, name2 = node.text.split()

            name1 = name1.strip('@')
            if name1 not in SAMOVAR_TEAM:
                raise Exception("Found Unknown tier-1 login: %s" % name1)
            duties['tier-1'] = Person.create(name1, SAMOVAR_TEAM[name1])

            name2 = name2.strip('@')
            if name2 not in SAMOVAR_TEAM:
                raise Exception("Found Unknown tier-2 login: %s" % name2)
            duties['tier-2'] = Person.create(name2, SAMOVAR_TEAM[name2])
            break

        return duties

    def create_mail(self, duties, receivers):
        text = "Hello,\nNew duty week has started. Tier1 duty is %s and Tier2 duty is %s.\nPlease read the guide:\nhttps://wiki.yandex-team.ru/robot/samovar/duty/" % \
                    (duties['tier-1'].Login, duties['tier-2'].Login)

        msg = email.mime.text.MIMEText(text)
        msg['Subject'] = "Samovar duties"
        msg['From'] = 'robot-samovar-duty@yandex-team.ru'
        msg['To'] = ",".join([r + "@yandex-team.ru" for r in receivers])
        return msg.as_string()

    def on_execute(self):
        duties = self.get_current_duties()
        logging.info("Found current duties for tiers: %s" % str(duties.keys()))
        for tier, person in duties.items():
            logging.info("Set %s as %s duty" % (person.Login, tier))
            self.set_on_duty(self.Robots[tier], person)

        self.update_juggler_rule(JUGGLER_RULE_ID, duties)

        try:
            srv = smtplib.SMTP('yabacks.yandex.ru', port=25)
            receivers = ["%s@yandex-team.ru" % login for login in ['the_samovar', duties['tier-1'].Login, duties['tier-2'].Login]]
            srv.sendmail('ArrangeSamovarDuties sandbox task <robot-samovar-duty@yandex-team.ru>', receivers, self.create_mail(duties, receivers))
            srv.quit()
        except smtplib.SMTPException, e:
            raise Exception('No e-mails was sent. Internal SMTP exception occured: %s' % str(e))


if __name__ == "__main__":
    logging.basicConfig(format=LOG_FORMAT, level=logging.INFO)
    a = ArrangeSamovarDuties()
    a.on_execute()
