import re
import os
import socket
import logging

from os.path import dirname, join
from sandbox.sandboxsdk.process import run_process  # , kill_process


class YTRunner(object):
    def __init__(self, yt_wrapper, yt_bin, log_dir):
        self.wrapper = yt_wrapper
        self.yt_bin = yt_bin
        self.log_dir = log_dir
        self.id = None
        self.proxy = None
        self.port = None
        self.package_dir = dirname(dirname(self.wrapper))
        logging.debug("YT package_dir: {}".format(self.package_dir))

        env = dict(os.environ)
        yt_env = {
            "NODE_PATH": join(self.package_dir, "node_modules"),
            "PYTHONPATH": join(self.package_dir, "python"),
            "PATH":  "{0}/node/bin:{0}/bin".format(self.package_dir),
            "YT_LOCAL_THOR_PATH": join(self.package_dir, "yt-thor")
        }

        logging.debug("PYTHONPATH FILES:")
        for root, directories, files in os.walk(yt_env['PYTHONPATH']):
            for filename in files:
                logging.debug("     {}".format(os.path.join(root, filename)))

        yt_env = {k: str(v) for k, v in yt_env.iteritems()}
        for k, v in yt_env.iteritems():
            if k in env and env[k]:
                env[k] = "{}:{}".format(v, env[k])
            else:
                env[k] = v
        self.env = env
        logging.debug("self.env type: {}".format(type(self.env)))
        logging.debug("NODE_PATH: {}".format(self.env['NODE_PATH']))
        logging.debug("PYTHONPATH: {}".format(self.env['PYTHONPATH']))
        logging.debug("PATH: {}".format(self.env['PATH']))
        logging.debug("YT_LOCAL_THOR_PATH: {}".format(self.env['YT_LOCAL_THOR_PATH']))

    def start(self, rpc, cypress_dir=None, additional_params=None):
        stdout_file = join(self.log_dir, 'yt_local.start.out')
        stderr_file = join(self.log_dir, 'yt_local.start.err')
        stdout = open(stdout_file, 'w')
        stderr = open(stderr_file, 'w')
#        cmd = [self.wrapper, 'start', '--enable-debug-logging', '--proxy-port', '26324']
        cmd = [self.wrapper, 'start', '--enable-debug-logging']

        if (rpc):
            cmd.append('--rpc-proxy')

        if (cypress_dir):
            cmd.append('--local-cypress-dir')
            cmd.append(cypress_dir)

        if additional_params:
            cmd.extend(additional_params)

        try:
            run_process(cmd, stdout=stdout, stderr=stderr, environment=self.env)
            stderr.close()
            stderr = open(stderr_file)
            for l in stderr:
                m = re.match(r'.*proxy\s+addresses:\s+\[\'([^:]+):(\d+)', l)
                if m:
                    self.proxy = m.group(1)
                    self.port = m.group(2)
                m = re.match(r'.*id:\s+([0-9_a-z]{,8}-[0-9_a-z]{,8}-[0-9_a-z]{,8}-[0-9_a-z]{,8}).*', l)
                if m:
                    self.id = m.group(1)
            if self.id is None:
                raise Exception("Couldn't get session ID from: {}".format(stderr_file))
        finally:
            stdout.close()
            stderr.close()

    def stop(self):
        stdout_file = join(self.log_dir, 'yt_local.stop.out')
        stderr_file = join(self.log_dir, 'yt_local.stop.err')
        stdout = open(stdout_file, 'w')
        stderr = open(stderr_file, 'w')
        cmd = [self.wrapper, 'stop', self.id]

        try:
            run_process(cmd, stdout=stdout, stderr=stderr, environment=self.env)
        finally:
            stdout.close()
            stderr.close()

    def set_acl(self, where, what):
        stdout_file = join(self.log_dir, 'yt_local.set_acl.out')
        stderr_file = join(self.log_dir, 'yt_local.set_acl.err')
        stdout = open(stdout_file, 'w')
        stderr = open(stderr_file, 'w')
        cmd = [self.yt_bin, '--proxy', "{}:{}".format(self.get_proxy(), self.get_port()), 'set', '{}/@acl'.format(where), "{}".format(what)]
        try:
            run_process(cmd, stdout=stdout, stderr=stderr, environment=self.env)
        finally:
            stdout.close()
            stderr.close()

    def create_user(self, user_name):
        stdout_file = join(self.log_dir, 'yt_local.create_user.out')
        stderr_file = join(self.log_dir, 'yt_local.create_user.err')
        stdout = open(stdout_file, 'w')
        stderr = open(stderr_file, 'w')
        cmd = [self.yt_bin, "--proxy", "{}:{}".format(self.get_proxy(), self.get_port()), "create", "user", "--attributes", "{name="+user_name+"}"]
        try:
            run_process(cmd, stdout=stdout, stderr=stderr, environment=self.env)
        finally:
            stdout.close()
            stderr.close()

    def get_rpc_proxy(self):
        stdout_file = join(self.log_dir, 'yt_local.list.out')
        stderr_file = join(self.log_dir, 'yt_local.list.err')
        stdout = open(stdout_file, 'w')
        stderr = open(stderr_file, 'w')

        cmd = [self.yt_bin, "--proxy", "{}:{}".format(self.get_proxy(), self.get_port()), "list",  "//sys/rpc_proxies"]
        try:
            run_process(cmd, stdout=stdout, stderr=stderr, environment=self.env)
        finally:
            stdout.close()
            stderr.close()
            with open(stdout_file) as f:
                for l in f:
                    m = re.search(r'[^:]+:(\d+)', l)
                    if m:
                        return "{}:{}".format(socket.gethostname(), m.group(1))

        return None

    def list(self, path):
        stdout_file = join(self.log_dir, 'yt_local.list.base.out')
        stderr_file = join(self.log_dir, 'yt_local.list.base.err')
        stdout = open(stdout_file, 'w')
        stderr = open(stderr_file, 'w')

        cmd = [self.yt_bin, "--proxy", "{}:{}".format(self.get_proxy(), self.get_port()), "list",  path]
        try:
            run_process(cmd, stdout=stdout, stderr=stderr, environment=self.env)
        finally:
            stdout.close()
            stderr.close()
            with open(stdout_file) as f:
                for l in f:
                    logging.debug("BASE TABLE: {}".format(l))

    def delete(self):
        stdout_file = join(self.log_dir, 'yt_local.delete.out')
        stderr_file = join(self.log_dir, 'yt_local.delete.err')
        stdout = open(stdout_file, 'w')
        stderr = open(stderr_file, 'w')
        cmd = [self.wrapper, 'delete', self.id]

        try:
            run_process(cmd, stdout=stdout, stderr=stderr, environment=self.env)
        finally:
            stdout.close()
            stderr.close()

    def get_proxy(self):
        return self.proxy

    def get_port(self):
        return self.port

    def get_id(self):
        return self.id
