import os
import re
import logging

import sandbox.projects.samovar as samovar_resources
from os.path import join as pjoin
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.projects.common.async_res_download import AsyncResDownloadTask


class SamovarTestCompare(AsyncResDownloadTask):
    type = 'SAMOVAR_TEST_COMPARE'

    class LeftCompareResource(ResourceSelector):
        name = 'left_resource'
        description = 'Left resource to compare'
        resource_type = samovar_resources.SAMOVAR_TEST_OUT
        required = True

    class RightCompareResource(ResourceSelector):
        name = 'right_resource'
        description = 'Right resource to compare'
        resource_type = samovar_resources.SAMOVAR_TEST_OUT
        required = True

    input_parameters = [LeftCompareResource, RightCompareResource]

    RES_KEYS_ASYNC_DOWNLOAD = [LeftCompareResource.name, RightCompareResource.name]

    EXCLUDED_ITEMS = {"tender": {'ZLogCounters': '*',
                                 'Uptime': ''
                                },
                      "combustor": {'Uptime': '',
                                    'NoTablet': '',
                                    'ZLogCounters': '*',
                                    'TransactionsNoPushes': '',
                                    'QueueLatency': '*',
                                    'TransactionLatency': '*',
                                    'LatencyStartTx': '*',
                                    'LatencyReadPushes': '*',
                                    'LatencyLoadData': '*',
                                    'LatencyWork': '*',
                                    'AmountUrlPushesPerTx': '*',
                                    'AmountHostPushesPerTx': '*',
                                    'LatencyCommitTx': '*',
                                    'TransactionsSucceeded': '',
                                    'HostRecordsRead': '',
                                    'HostRecordsDataWritten': '',
                                    'HostRecordsDataDiffWritten': '',
                                    'HostRecordsModified': ''
                                   }
                     }

    def _get_resource_revision(self, resource_id):
        GET_RESOURCE_REVISION_RE = re.compile(r'.*Revision\s*:\s*(\d+)')
        resource_description = channel.sandbox.get_resource(resource_id).description
        m = GET_RESOURCE_REVISION_RE.search(resource_description)
        if m:
            return m.group(1)
        return None

    def _write_file(self, f, content):
        with open(f, 'w') as F:
            for i in content:
                F.write(i)

    def _get_file_content(self, f):
        content = []
        with open(f, 'r') as F:
            content = F.readlines()
        return content

    def _is_excluded_value(self, item, list):
        for k in list.keys():
            if k in item:
                return list[k]
        return None

    def _filter_file_content(self, content, fileName):
        res = []
        excList = None
        bracketsCount = 0

        for k in self.EXCLUDED_ITEMS.keys():
            if k in fileName:
                excList = self.EXCLUDED_ITEMS[k]

        if not excList:
            return None

        for i in content:
            if bracketsCount == 0:
                excluded = self._is_excluded_value(i, excList)
                if excluded == '' and bracketsCount == 0:
                    continue
                elif excluded == '*' and bracketsCount == 0 and '{' in i:
                    bracketsCount += 1
                    continue
                else:
                    res.append(i)
            else:
               if '{' in i:
                   bracketsCount += 1
               elif '}' in i:
                   bracketsCount -= 1
        return res

    def on_execute(self):
        work_dir = os.getcwd()
        left_work_dir = pjoin(work_dir, 'left')
        right_work_dir = pjoin(work_dir, 'right')
        left_pro_dir = pjoin(work_dir, 'left_pro')
        right_pro_dir = pjoin(work_dir, 'rith_pro')

        make_folder(left_work_dir)
        make_folder(right_work_dir)
        make_folder(left_pro_dir)
        make_folder(right_pro_dir)

        logging.debug("Left compared resource :{}".format(self.ctx[self.LeftCompareResource.name]))
        logging.debug("Right compared resource :{}".format(self.ctx[self.RightCompareResource.name]))

        left_resource_revision = self._get_resource_revision(self.ctx[self.LeftCompareResource.name])
        if left_resource_revision is None:
            raise SandboxTaskFailureError("Couldn't get revision for left resource")

        right_resource_revision = self._get_resource_revision(self.ctx[self.RightCompareResource.name])
        if right_resource_revision is None:
            raise SandboxTaskFailureError("Couldn't get revision for right resource")

        logging.debug("Left resource revision: {}".format(left_resource_revision))
        logging.debug("Right resource revision: {}".format(right_resource_revision))

        procs = self.start_async_download()

        left_resource_dir = self.get_results_from_special_proc(procs, self.LeftCompareResource.name)
        right_resource_dir = self.get_results_from_special_proc(procs, self.RightCompareResource.name)

        logging.debug("Left resource dir: {}".format(left_resource_dir))
        logging.debug("Right resource dir: {}".format(right_resource_dir))

        n1 = len(list(os.listdir(left_resource_dir)))
        n2 = len(list(os.listdir(right_resource_dir)))
        if n1 != n2:
            raise SandboxTaskFailureError("Different files number in compared resources:"
                                          " {} VS {}".format(n1, n2))

        for f in os.listdir(left_resource_dir):
            cmd = ['tar', '-xzf', pjoin(left_resource_dir, f), '-C', left_work_dir]
            run_process(cmd, check=True, log_prefix=f+'.untar', outputs_to_one_file=True, wait=True)
            cmd = ['tar', '-xzf', pjoin(right_resource_dir, f), '-C', right_work_dir]
            run_process(cmd, check=True, log_prefix=f+'.untar', outputs_to_one_file=True, wait=True)

        for f in os.listdir(left_work_dir):
            left_file_content = self._get_file_content(pjoin(left_work_dir, f))
            right_file_content = self._get_file_content(pjoin(right_work_dir, f))

            left_file_content = self._filter_file_content(left_file_content, f)
            right_file_content = self._filter_file_content(right_file_content, f)

            self._write_file(pjoin(left_pro_dir, f), left_file_content)
            self._write_file(pjoin(right_pro_dir, f), right_file_content)

        for f in os.listdir(left_pro_dir):
            cmd = ['diff', '-uNr', pjoin(left_pro_dir, f), pjoin(right_pro_dir, f)]
            run_process(cmd, check=True, log_prefix=f, outputs_to_one_file=True, wait=True)


__Task__ = SamovarTestCompare
