import logging

from sandbox.common import rest
from sandbox.common import auth
from sandbox.common import errors
from sandbox.common.types import task as ctt

from sandbox import sdk2

DEFAULT_KEY = "sandbox-oauth-token"


class PreprodChecker(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        preprod_oauth_token = sdk2.parameters.YavSecret(
            "PreProd Sandbox oauth token",
            default="sec-01dkf3bkynzvmghts08j3r8q8v#{}".format(DEFAULT_KEY),
            required=True,
        )
        target_task_type = sdk2.parameters.String(
            "Target Task type",
            default="SANDBOX_ACCEPTANCE",
            required=True,
        )
        owner_group = sdk2.parameters.String(
            "Group to run target task",
            default="SANDBOX",
            required=True,
        )
        init_wait_time_minutes = sdk2.parameters.Integer(
            "Sleeping time (minutes)",
            default=10,
            required=True
        )

    def _get_timeout(self, step=0):
        init_time = self.Parameters.init_wait_time_minutes
        wait_time_seconds = {
            0: 1 * init_time * 60,
            1: 0.5 * init_time * 60,
            2: 0.3 * init_time * 60,
        }
        return wait_time_seconds.get(step, 60)

    def _validate_target_task_id(self, preprod_task_id):
        if preprod_task_id is not None:
            self.Context.preprod_task_id = preprod_task_id
            return
        raise errors.TaskFailure(
            "Couldn't run task '{}' using '{}'".format(
                self.Parameters.target_task_name,
                self.Parameters.owner
            )
        )

    def _get_saved_task(self, api_preprod):
        task_id = self.Context.preprod_task_id
        task_description = api_preprod.task[task_id].read()
        task_status = task_description["status"]

        return task_id, task_status

    def on_execute(self):
        secret_key = self.Parameters.preprod_oauth_token.data()[
            self.Parameters.preprod_oauth_token.default_key or DEFAULT_KEY
        ]
        oauth = auth.OAuth(secret_key)
        api_preprod = rest.Client(
            base_url="https://www-sandbox1.n.yandex-team.ru/api/v1.0",
            auth=oauth,
            total_wait=60,
        )
        logging.debug("PreProd API client is ready")

        with self.memoize_stage.first_step():
            task_id = api_preprod.task(
                type=self.Parameters.target_task_type,
                owner=self.Parameters.owner_group,

            ).get("id")

            self._validate_target_task_id(task_id)
            api_preprod.batch.tasks.start.update({"id": task_id})

            logging.info("Initial step, task '%d' created", task_id)
            self.set_info(
                '<p>'
                'Preprod task '
                '#<a href="https://sandbox-prestable.yandex-team.ru/task/{}/view">{}</a> '
                'started'
                '</p>'.format(task_id, task_id),
                do_escape=False
            )

            raise sdk2.WaitTime(
                self._get_timeout()
            )

        with self.memoize_stage.check_task_status(100) as counter:
            logging.info("Step #%d", counter.runs)
            task_id, task_status = self._get_saved_task(api_preprod)
            logging.info("Got task_id '%d' with status '%s'", task_id, task_status)

            if task_status in ctt.Status.Group.BREAK:
                raise errors.TaskFailure(
                    "PreProd task '{}' broke with status: '{}'".format(task_id, task_status)
                )
            elif task_status in ctt.Status.Group.FINISH:
                if task_status == ctt.Status.SUCCESS:
                    return
                else:
                    raise errors.TaskFailure(
                        "PreProd task '{}' failed with status: '{}'".format(task_id, task_status)
                    )

            raise sdk2.WaitTime(
                self._get_timeout(counter.runs)
            )

        with self.memoize_stage.last_step(1):
            logging.info("Final step")
            task_id, task_status = self._get_saved_task(api_preprod)

            if task_status == ctt.Status.SUCCESS:
                return
            raise errors.TaskError(
                "Couldn't wait for the task '{}' to finish in 100 runs. Last status is '{}'".format(
                    task_id,
                    task_status
                )
            )
