# coding: utf-8
from __future__ import unicode_literals

import json
import logging
import collections

import six

from sandbox import sdk2
from sandbox.sdk2.vcs import svn

from sandbox.common import rest
from sandbox.common import errors
from sandbox.common import format as cformat
from sandbox.common.types import task as ctt
from sandbox.common.types import client as ctc
from sandbox.common.types import resource as ctr

Resource = collections.namedtuple("Resource", ("id", "size"))


class DevtoolsResourceCollector(sdk2.Resource):
    executable = True
    releasable = True
    arcadia_build_path = "sandbox/scripts/collect_devtools_resources"


class DevtoolsResources(sdk2.Resource):
    pass


class BackupDevtoolsResources(sdk2.Task):
    """
    The task ensures that resources critital for devtools are backed up in at least two strorages in different DCs.
    """

    class Requirements(sdk2.Requirements):
        disk_space = 20 * 1024

    class Parameters(sdk2.Parameters):
        description = "Backup resources critical for devtools"
        kill_timeout = 3600
        resource_list = sdk2.parameters.Resource(
            "Resources for backup", required=False, resource_type=DevtoolsResources,
            description="Leave blank to fetch critical resources automatically"
        )

    def on_enqueue(self):
        if not self.Requirements.tasks_resource:
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(attrs={
                "released": ctt.ReleaseStatus.STABLE,
                "task_type": self.type.name,
            }).first()
        if not self.Requirements.tasks_resource:
            raise errors.TaskError("Binary task resource is not found")

    def on_execute(self):

        if self.Parameters.resource_list:
            # Server-side

            rd = sdk2.ResourceData(self.Parameters.resource_list)
            with rd.path.open("r") as f:
                resource_ids = json.load(f)

            need_backup = self.need_backup(resource_ids)

            with self.log_path("need_backup.json").open("wb") as f:
                json.dump(need_backup, f, indent=2)

            self.run_backup_tasks(need_backup)
            return

        # Client-side

        with self.memoize_stage.fetch(commit_on_entrance=False):

            arcadia_src_dir = svn.Arcadia.get_arcadia_src_dir(svn.Arcadia.trunk_url())
            if not arcadia_src_dir:
                raise errors.TaskFailure("Could not fetch Arcadia")

            collector_resource = DevtoolsResourceCollector.find(
                state=ctr.State.READY, attrs={"released": ctt.ReleaseStatus.STABLE}
            ).first()
            if not collector_resource:
                raise errors.ReleaseError("Could not find released {} resource".format(DevtoolsResourceCollector.name))
            collector = sdk2.ResourceData(collector_resource)

            with sdk2.helpers.ProcessLog(self, logger="collector") as pl:
                output = sdk2.helpers.subprocess.check_output([str(collector.path), arcadia_src_dir], stderr=pl.stderr)
            resources = json.loads(output)  # sanity check for valid JSON

            res = DevtoolsResources(
                task=self,
                description="list of critical resources",
                path="./resources.json"
            )
            rd = sdk2.ResourceData(res)
            with rd.path.open("wb") as f:
                json.dump(resources, f)
            rd.ready()

        with self.memoize_stage.run_backup_tasks(commit_on_entrance=False):
            task = BackupDevtoolsResources(
                self, owner=self.owner, resource_list=res.id,
                __requirements__={
                    "client_tags": ctc.Tag.SERVER,
                    "tasks_resource": self.Requirements.tasks_resource,
                }
            )
            task.enqueue()
            raise sdk2.WaitTask([task.id], [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK], wait_all=True)

    @staticmethod
    def need_backup(resource_ids):
        """
        :param resource_ids: List[int]
        :rtype: Dict[str, List[Resource]]
        """
        from sandbox.yasandbox.database import mapping
        mapping.ensure_connection()

        h2dc = {_.hostname: _.info["system"]["dc"] for _ in mapping.Client.objects.all()}
        r2dcs = {}
        query = (
            mapping.Resource.objects(id__in=resource_ids, state="READY", mds__eq=None)
            .fast_scalar("id", "hosts_states", "attributes", "size")
        )

        for rid, hosts, attrs, size in query:
            if not any(a.get("k") == "backup_task" or a == {"k": "ttl", "v": "inf"} for a in attrs or []):
                continue
            r2dcs[Resource(rid, size)] = {
                h2dc[h['h']] for h in hosts or [] if h.get('h') in h2dc and h.get("st") == "OK"
            }

        resources = collections.defaultdict(list)

        for res, dcs in r2dcs.items():
            if len(dcs) == 1:
                resources[list(dcs)[0]].append(res)

        return resources

    def run_backup_tasks(self, resources):

        futures = []
        with rest.Batch(self.server) as batch_client:
            for avoid_dc, resources in six.iteritems(resources):
                disk_space = sum(res.size for res in resources) << 10  # bytes

                futures.append(batch_client.task(
                    children=self.parent.id if self.parent is not None else self.id,
                    type="BACKUP_RESOURCE_2",
                    description=(
                        "Regular backup of critical resources with total size of {}."
                        .format(cformat.size2str(disk_space))
                    ),
                    owner=self.owner,
                    requirements={"disk_space": int(disk_space * 1.5)},
                    notifications=[],
                    custom_fields=[
                        {"name": "resource_id", "value": ",".join(str(res.id) for res in resources)},
                        {"name": "avoid_dc", "value": avoid_dc},
                    ],
                    priority={"class": "USER", "subclass": "HIGH"},
                ))

        tasks = [f.result()["id"] for f in futures]

        logging.info("Subtasks: %s", tasks)
        self.server.batch.tasks.start.update({"id": tasks})

        return tasks
