import six
import collections

import os
import logging
import threading

from sandbox import common
from sandbox import sdk2
from sandbox.agentr import errors as ar_errors
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

from sandbox.projects.common import binary_task


class BackupResource2(sdk2.ServiceTask, binary_task.LastBinaryTaskRelease):
    """
    Backup resources to the machine
    "Resource" field - string with comma-separated resource ids
    These resources are downloaded to the machine.
    This task is usually run on sandbox-storageX machines
    """
    temporary_error = False
    in_queue = collections.deque()

    class Tag(common.utils.Enum):
        AUTOBACKUP = None
        REDUNDANCY = None

    class Parameters(sdk2.Task.Parameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        IGNORE_LOW_DISK_SERVICE_GROUPS = ("SANDBOX",)  #: Service groups, which are allowed to ignore low disk space tag

        resource_id = sdk2.parameters.String("Resources", required=False)
        register_deps = sdk2.parameters.Bool("Register backed up resources as the task deps", default=True)
        validate_backup_task = sdk2.parameters.Bool("Validate backup task ID (service only)", default=False)
        try_restore = sdk2.parameters.Bool("Attempt to restore BROKEN resources (admin only)", default=False)
        thread_pool_size = sdk2.parameters.Integer("Thread pool size", default=9, required=True)
        redundancy_semaphore = sdk2.parameters.Bool("Use redundancy semaphore to limit number of tasks", ui=None)
        ignore_low_disk = sdk2.parameters.Bool("Ignore `CUSTOM_LOW_DISK` tag")
        source = sdk2.parameters.String("Source host (for redundancy tasks, for search only)")
        chunk_size = sdk2.parameters.Integer("Chunk size", default=0)
        replication = sdk2.parameters.Bool("Replication", default=False)
        dc = sdk2.parameters.String("DC", default=None)
        avoid_dc = sdk2.parameters.String("Avoid DC", default=None)

    class Requirements(sdk2.Requirements):
        client_tags = (
            ctc.Tag.GENERIC | ctc.Tag.STORAGE | ctc.Tag.CUSTOM_REPLICATION | ctc.Tag.CUSTOM_LOW_DISK
        ) & ctc.Tag.HDD

    class Context(sdk2.Task.Context):
        fail_queue = []

    def validate_resource_ids(self):
        resource_ids = [int(r_id.strip()) for r_id in self.Parameters.resource_id.split(",") if r_id.strip()]
        return resource_ids

    @staticmethod
    def shard_files(path):
        files = ["shard.conf"]
        with open(os.path.join(path, files[0]), "r") as shard_conf:
            for line in shard_conf:
                if line.startswith("%attr"):
                    files.append(line.split()[-1])
        return files

    def reshare_shard(self, resource, path):
        try:
            files = self.shard_files(path)
        except (OSError, IOError) as ex:
            logging.warning("Unable to share resource #%s as a shard: %s", resource["id"], ex)
            return

        logging.debug("Files specified in shard.conf: %r", files)
        try:
            skyid = common.share.ShareSandboxResource(None, files, path).run()
        except Exception:
            logging.exception("Unable to share resource #%s as a shard", resource["id"])
            return
        logging.info("Reshared resource %s (%s) as a shard with skynet ID %s", resource["id"], resource["type"], skyid)

    def sync_resource(self, resource):
        try:
            # FIXME: SANDBOX-8967: Temporary disabled to backup resources with multiple "backup_task" attributes in the database
            if False and self.Parameters.validate_backup_task and int(resource["attributes"]["backup_task"]) != self.id:
                return
        except (KeyError, ValueError):
            pass
        if self.Parameters.try_restore:
            path = self.agentr.resource_sync(resource["id"], restore=True)
        else:
            # TODO: SANDBOX-6941: Can be collapsed with block above after AgentR upgrade.
            path = self.agentr.resource_sync(resource["id"])
        try:
            if sdk2.Resource[resource["type"]].shard:
                try:
                    self.reshare_shard(resource, path)
                except Exception:
                    logging.exception("Failed to reshare shard")
                    raise
        except common.errors.UnknownResourceType as ex:
            logging.warning(ex)

        return path

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)

        if self.Parameters.redundancy_semaphore:
            self.Requirements.semaphores = ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(name="sandbox/backup_resource/redundancy", capacity=3)
                ]
            )

    def on_enqueue(self):
        import yasandbox.manager.task as tm
        autobackup = "AUTOBACKUP" in self.Parameters.tags
        self.Requirements.client_tags = (
            ctc.Tag.GENERIC
            if "FORCE" in self.Parameters.tags and autobackup else
            ctc.Tag.STORAGE | ctc.Tag.CUSTOM_REPLICATION | ctc.Tag.CUSTOM_LOW_DISK
        )
        if not autobackup:  # We cannot force specific DC for autobackup tasks 'cause we haven't storage hosts in each DC
            if self.Parameters.dc:
                self.Requirements.client_tags &= ctc.Tag[self.Parameters.dc.upper()]
            if self.Parameters.avoid_dc:
                self.Requirements.client_tags &= ~ctc.Tag[self.Parameters.avoid_dc.upper()]

        resource_ids = self.Context.fail_queue or self.validate_resource_ids()
        if self.Parameters.register_deps:
            for resource_id in resource_ids:
                tm.TaskManager.register_dep_resource(self.id, resource_id)
        self.Context.original_resource_ids = resource_ids[:]

    def worker_loop(self):
        counter = 0
        valid_states = (
            ctr.State.READY,
            ctr.State.BROKEN
        )
        name = threading.current_thread().name
        logging.debug("Worker %s started.", name)
        while True:
            try:
                # Resources should be sorted by size descending, so try to get bigger resources first.
                resource_id = self.in_queue.popleft()
                logging.info("Worker %s: processing resource #%s.", name, resource_id)
                counter += 1
            except IndexError:
                break
            try:
                resource = self.server.resource[resource_id][:]
            except common.rest.Client.HTTPError:
                continue
            if not resource or resource["state"] not in valid_states:
                continue
            try:
                self.sync_resource(resource)
            except Exception as e:
                self.Context.fail_queue.append(resource_id)
                self.set_info("Can't backup resource {}:\n{}".format(resource_id, str(e) or repr(e)))
                if isinstance(e, ar_errors.NoSpaceLeft):
                    self.temporary_error = True
        logging.debug("Worker %s stopped. Totally %s resource(s) processed.", name, counter)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        resource_ids = self.Context.fail_queue or self.validate_resource_ids()
        pool_size = min(len(resource_ids), self.Parameters.thread_pool_size)
        logging.debug("Thread pool size: %d", pool_size)

        self.in_queue = collections.deque(resource_ids)
        self.Context.fail_queue = []

        if pool_size > 1:
            pool = [threading.Thread(target=self.worker_loop, name="#%d" % i) for i in six.moves.range(pool_size)]
            list(map(threading.Thread.start, pool))
            logging.debug("Waiting for all workers to finish.")
            list(map(threading.Thread.join, pool))
        else:
            self.worker_loop()
        self.Context.fail_queue = [r_id for r_id in resource_ids if r_id in self.Context.fail_queue]

        if len(self.Context.fail_queue):
            raise (
                common.errors.TemporaryError
                if self.temporary_error else
                common.errors.TaskError
            )("Unable to backup {} resource(s).".format(len(self.Context.fail_queue)))
