# coding: utf-8

import shutil
import os.path
import subprocess as sp

import pathlib2

from sandbox import common, sdk2
from sandbox.common.types import task as ctt
from sandbox.common.types import client as ctc
from sandbox.common.types import notification as ctn


class ServiceqSnapshot(sdk2.Resource):
    """
    ServiceQ snapshot
    """
    auto_backup = True
    ttl = 5


class BackupServiceq(sdk2.Task):
    """
    Save the most recent ServiceQ snapshot as a resource
    """

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.SERVER
        disk_space = 150

    class Parameters(sdk2.Parameters):
        description = "Backup ServiceQ"
        kill_timeout = 10 * 60
        notifications = (
            sdk2.Notification(
                (ctt.Status.Group.BREAK, ctt.Status.FAILURE),
                ("sandbox-errors",),
                ctn.Transport.EMAIL
            ),
        )

    def on_execute(self):
        settings = common.config.Registry()
        runtime_dir = pathlib2.Path(settings.common.dirs.runtime)

        files = list(runtime_dir.glob("serviceq.*.snapshot"))
        if not files:
            raise common.errors.TaskFailure("Snapshot files not found")

        snapshot_source = max(files, key=lambda p: p.stat().st_mtime)  # get the most recent file
        snapshot_source = str(snapshot_source)

        snapshot_backup = str(self.path(os.path.basename(snapshot_source)))

        shutil.copy2(snapshot_source, snapshot_backup)

        with sdk2.helpers.ProcessLog(self, logger="compression") as pl:
            sp.check_call(["xz", "-9", snapshot_backup], stderr=pl.stderr, stdout=pl.stdout)

        compressed_backup = snapshot_backup + ".xz"
        description = "ServiceQ snapshot. Decompress with `unxz {}`".format(os.path.basename(compressed_backup))

        resource = ServiceqSnapshot(self, description, compressed_backup)
        sdk2.ResourceData(resource).ready()
