import os
import yaml
import logging
import functools as ft

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

from sandbox import common
import sandbox.common.types.client as ctc


class MetadataBackup(sdk2.Resource):
    """
    A tarball containing skynet.copier and sandbox.agentr databases, together with /etc/fstab
    """

    auto_backup = True
    hostname = sdk2.parameters.String("FQDN of the host this backup belongs to")


class BackupStorageMetadata(sdk2.ServiceTask):
    """
    Back up skynet.copier and sandbox.agentr databases, together with /etc/fstab (SANDBOX-3265).
    To restore saved metadata, use scripts/restore_storage_metadata.py
    """

    class Requirements(sdk2.Requirements):
        execution_space = 30 * 1024  # 30 Gb should suffice

    def on_execute(self):
        if self.parent is None:  # running as a scheduler
            query = ft.partial(self.server.client.read, tags=str(ctc.Tag.STORAGE & ctc.Tag.NEW_LAYOUT), alive=True)
            limit = query(limit=0)["total"]
            children = []
            for host in query(limit=limit)["items"]:
                child = BackupStorageMetadata(
                    self,
                    description="Metadata backup for {}".format(host["id"]),
                )
                child.Requirements.host = host["id"]
                children.append(child.save().id)
                logging.debug("Child with id %d is set for execution @ %s", child.id, host["id"])
            if children:
                self.server.batch.tasks.start.update(id=children)
            return

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("skybone-dbbackup")) as pl:
            sp.Popen(["/skynet/tools/skybone-ctl", "dbbackup"], stdout=pl.stdout, stderr=sp.STDOUT).wait()
        skynet_cfg = yaml.load(open("/skynet/.info", "r"))

        paths = {
            "agentr.db": self.agentr.backup(),
            "copier.db": os.path.join(
                skynet_cfg["paths"]["prefix"], "supervisor/var/copier/rbtorrent/copier.db.backup"
            ),
            "fstab": "/etc/fstab"
        }
        logging.debug("%r", paths)

        tarname = "backup.tar.pixz"
        with sdk2.helpers.ProcessLog(self, logger="tar") as pl:
            rc = sp.Popen(
                ["tar", "-Ipixz", "-cf", tarname] + paths.values(),
                stdout=pl.stdout, stderr=sp.STDOUT
            ).wait()
            if rc:
                raise common.errors.TaskFailure("Failed to compress backups")

        sdk2.ResourceData(
            MetadataBackup(
                self,
                "Metadata backup for {}".format(self.host),
                tarname,
                hostname=self.Requirements.host
            )
        ).ready()
