from __future__ import absolute_import

import os
import tarfile
import contextlib

from sandbox import common

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox.sdk2.helpers.process import subprocess as sp

from sandbox import sandboxsdk

import sandbox.projects.sandbox.resources as sb_resources


class BuildEventProcessor(sdk2.Task):
    class Requirements(sdk2.Requirements):
        # client_tags = ctc.Tag.LINUX_XENIAL  # It should match server's platform for STEP
        client_tags = ctc.Tag.LINUX_PRECISE
        disk_space = 2048
        dns = ctm.DnsType.DNS64
        privileged = True

    class Parameters(sdk2.Parameters):
        description = "Build STEP"
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia URL for Sandbox", default_value="arcadia:/arc/trunk/arcadia/sandbox"
        )
        run_tests = sdk2.parameters.Bool("Run tests", default=True)

    def on_execute(self):
        import pip.req
        import pip.download

        resource_data = sdk2.ResourceData(sb_resources.EventProcessor(self, "Event processor (STEP)", "packages"))
        packages_path = resource_data.path
        packages_path.mkdir()

        with sdk2.helpers.ProgressMeter("Installing dependencies"):
            sp.check_call(["/usr/bin/apt-get", "update"])
            for package in ("libssl-dev", "libffi-dev"):
                sp.check_call(["/usr/bin/apt-get", "install", "-yq", package])

        with sdk2.helpers.ProgressMeter("Getting source code from Arcadia"):
            sandbox_path = self.path("sandbox")
            sdk2.svn.Arcadia.export(self.Parameters.arcadia_url, str(self.path("sandbox")))
            step_path = sandbox_path.joinpath("step")
            common_path = sandbox_path.joinpath("common")
            etc_path = sandbox_path.joinpath("etc")
            init_path = sandbox_path.joinpath("__init__.py")
            with contextlib.closing(tarfile.open(str(packages_path.joinpath("step.tgz")), "w:gz")) as tar_file:
                for path in (step_path, common_path, etc_path, init_path):
                    tar_file.add(str(path), arcname=os.path.join(sandbox_path.name, path.name))

        with sdk2.helpers.ProgressMeter("Building virtual environment"):
            with sandboxsdk.environments.VirtualEnvironment(do_not_remove=True) as venv:
                # gevent need use_cflags_ldflags=False
                venv.pip("-vvv -r " + str(step_path.joinpath("requirements.txt")), use_cflags_ldflags=False)

                # HACK: rebuild uwsgi with LDFLAGS to set correct RPATH
                def venv_freeze(target_venv):
                    cmd = [
                        target_venv.executable,
                        "-us",
                        os.path.join(os.path.dirname(target_venv.executable), "pip"),
                        "freeze"
                    ]
                    req_filename = self.path("requirements.txt")
                    with open(str(req_filename), "wb") as f, open(os.devnull, "wb") as devnull:
                        sp.Popen(cmd, stderr=devnull, stdout=f).wait()
                    return list(pip.req.parse_requirements(str(req_filename), session=pip.download.PipSession()))

                reqs = venv_freeze(venv)
                uwsgi = next(iter(req for req in reqs if req.req.name == "uWSGI"), None)
                if not uwsgi:
                    raise common.errors.TaskFailure("uWSGI not found in VENV")

                venv.pip("-I {}{} --no-cache-dir".format(uwsgi.req.name, uwsgi.req.specifier))
                venv.pack_bundle(str(packages_path.joinpath("venv_step.tgz")))

        if self.Parameters.run_tests:
            with sdk2.helpers.ProgressMeter("Running tests"), sdk2.helpers.ProcessLog(self, logger="tests") as pl:
                with sdk2.helpers.ProgressMeter("Installing packages for testing"):
                    venv.pip(
                        "-vvv mock==1.0.1 pytest==2.9.1 pytest-pep8==1.0.6"
                        " pytest-flakes==1.0.1 pytest-cache==1.0 pytest-django==3.1.2"
                    )
                self.path(".log").mkdir()
                env = os.environ.copy()
                env["PYTHONPATH"] = str(sandbox_path)
                env["YENV_TYPE"] = "development"
                try:
                    sp.check_call(
                        [
                            os.path.join(os.path.dirname(venv.executable), "py.test"),
                            "--pep8", "--flakes", "-v", str(step_path)
                        ],
                        env=env, stdout=pl.stdout, stderr=pl.stderr
                    )
                except sp.CalledProcessError as ex:
                    raise common.errors.TaskFailure(ex)
