import copy
import inspect
import logging

from sandbox import common
from sandbox import sandboxsdk

from sandbox import projects
from sandbox.projects.sandbox import subtasks_runner


class BuildMeForAll(subtasks_runner.SubTasksRunner):
    """Build specified package for all currently available platforms"""

    type = "BUILD_ME_FOR_ALL"
    input_parameters = [subtasks_runner.TaskType]

    @staticmethod
    def collect_platforms():
        platforms = set()
        for cl in sandboxsdk.channel.channel.sandbox.list_clients():
            if not cl["alive_now"]:
                continue
            platforms.add(common.platform.get_platform_alias(cl["platform"]))
        return platforms

    def on_execute(self):
        task_cls = projects.TYPES[self.ctx[subtasks_runner.TaskType.name]].cls
        base = sandboxsdk.task.BuildForAllMode
        if base not in inspect.getmro(task_cls):
            raise sandboxsdk.errors.SandboxTaskFailureError(
                "Task {cls} should be inheritor of {base_cls} class".format(cls=task_cls, base_cls=str(base))
            )

        if not self.ctx.get("subtasks_ids"):
            logging.info("Creating subtasks of {} type".format(task_cls.type))
            subtasks = []
            resources = []
            ctx = copy.deepcopy(self.ctx)
            for platform in self.collect_platforms():
                current_resources = []
                for res_type, descr in task_cls.required_resources():
                    res = self.create_resource(
                        description=descr,
                        arch=platform,
                        resource_path=self.path(platform),
                        resource_type=res_type,
                    )
                    current_resources.append((res.id, res_type))
                ctx["_prepared_resources"] = current_resources
                subtasks.append(
                    self.create_subtask(
                        task_type=task_cls.type,
                        arch=platform,
                        description="{type_} for {plat}".format(type_=task_cls.type, plat=platform),
                        input_parameters=ctx,
                    ).id
                )
                resources.extend(current_resources)
            logging.info("Starting to enqueue created tasks")
            for task_id in subtasks:
                sandboxsdk.channel.channel.sandbox.server.enqueue_task(task_id)
            self.ctx["subtasks_ids"] = subtasks
            logging.info("Subtasks successfully enqueued: %s", ", ".join([str(i) for i in subtasks]))
            logging.info("Prepared resources ids: %s", ", ".join([str(i) for i, _ in resources]))
            self.wait_all_tasks_completed(subtasks)
        else:
            for task in self.list_subtasks(load=True):
                if not task.is_finished():
                    raise sandboxsdk.errors.SandboxTaskFailureError("Subtask #{} is not finished".format(task.id))
            logging.info("All subtasks finished.")
