import os
import shutil
import hashlib
import logging
import platform
import multiprocessing as mp

import requests

from sandbox import common
import sandbox.common.types.misc as ctm

from sandbox import sandboxsdk

from sandbox.projects import resource_types


class BuildNodeJSPackage(sandboxsdk.task.BuildForAllMode):
    """
    Build NodeJS binary package of specified version from sources.
    """

    class Version(sandboxsdk.parameters.SandboxStringParameter):
        name = "sources_version"
        description = "NodeJS sources version"
        default_value = "6.11.3"

    environment = [sandboxsdk.environments.GCCEnvironment(version="5.3.0")]
    input_parameters = [Version]
    type = "BUILD_NODEJS_PACKAGE"

    dns = ctm.DnsType.DNS64

    resource_type = str(resource_types.NODEJS_PACKAGE)

    @classmethod
    def required_resources(cls):
        return [(cls.resource_type, "NodeJS package for {plat}".format(plat=platform.platform()))]

    def on_execute(self):
        version = self.ctx[self.Version.name]
        srcname = "node-v{}".format(version)
        srcfile = srcname + ".tar.gz"
        checksum_url = "http://nodejs.org/dist/v{}/SHASUMS256.txt.asc".format(version)
        srcpkg_url = "http://nodejs.org/dist/v{}/{}".format(version, srcfile)

        logging.info("Looking for source package's checksum at '%s'.", checksum_url)
        r = requests.get(checksum_url)
        checksum = next(
            (
                parts[0] for parts in (
                    line.split() for line in r.text.split('\n')
                ) if len(parts) == 2 and parts[1] == srcfile
            ),
            None
        )
        if not checksum:
            raise sandboxsdk.errors.SandboxTaskFailureError(
                "Unable to find a checksum for source file '{}'".format(srcfile)
            )
        logging.debug("SHA256 checksum of file '%s' is '%s'.", srcfile, checksum)

        with self.current_action("Downloading sources package from '{}'".format(srcpkg_url)):
            r = requests.get(srcpkg_url, stream=True)
            sha256 = hashlib.sha256()
            with open(srcfile, 'wb') as f:
                for chunk in r.iter_content(chunk_size=4095):
                    if chunk:  # filter out keep-alive new chunks
                        sha256.update(chunk)
                        f.write(chunk)
                        f.flush()

        logging.debug("SHA256 of downloaded file '%s' is '%s'.", srcfile, sha256.hexdigest())
        if checksum != sha256.hexdigest():
            raise sandboxsdk.errors.SandboxTaskFailureError(
                "Downloaded file's '{}' checksum '{}' doesn't match expected '{}'.".format(
                    srcpkg_url, sha256.hexdigest(), checksum
                )
            )

        srcpath = self.abs_path(srcname)
        with self.current_action("Extracting sources to '{}'".format(srcpath)):
            if os.path.exists(srcpath):
                shutil.rmtree(srcpath)
            os.mkdir(srcpath)
            sandboxsdk.process.run_process(
                ["tar", "xf", srcfile, "--strip-components=1", "--directory", srcpath],
                log_prefix="extract"
            )

        with self.current_action("Building sources at '{}'".format(srcpath)):
            env = os.environ.copy()
            env["PYTHON"] = "/skynet/python/bin/python"
            builddir = os.path.join(srcpath, "build")
            sandboxsdk.process.run_process(
                [env["PYTHON"], "configure", "--prefix", builddir],
                work_dir=srcpath, log_prefix="configure"
            )
            sandboxsdk.process.run_process(
                ["make", "-j", str(mp.cpu_count()), "-C", srcpath],
                environment=env, log_prefix="build"
            )
            cmd = [env["PYTHON"], "tools/install.py", "install", builddir, ""]
            if self.ctx[self.Version.name].startswith("0.10."):
                cmd[-2] = ""
            sandboxsdk.process.run_process(
                cmd, environment=env, work_dir=srcpath, log_prefix="install"
            )

        with self.current_action("Embedding `npm`"):
            moddir = os.path.join(builddir, "lib", "node_modules")
            bindir = os.path.join(builddir, "bin")
            npm = os.path.join(bindir, "npm")

            common.fs.make_symlink(
                os.path.relpath(os.path.join(moddir, "npm", "bin", "npm-cli.js"), bindir), npm,
                force=True
            )

        with self.current_action("Packaging distributive"):
            distpkg = self.abs_path("node.tar.gz")
            sandboxsdk.process.run_process(
                ["tar", "-zcvf", distpkg, "-C", builddir, '.'],
                log_prefix="pack"
            )

        prepared_resources = self.prepared_resources()
        attrs = {"version": version, "platform": platform.platform()}
        if prepared_resources:
            id_ = prepared_resources[0][0]
            for k, v in attrs.iteritems():
                sandboxsdk.channel.channel.sandbox.set_resource_attribute(id_, k, v)
            self.save_parent_task_resource(distpkg, id_)
            logging.info("Resource #%s saved to parent task", id_)
        else:
            res = self.create_resource(
                description=self.required_resources()[0][1],
                resource_path=distpkg,
                resource_type=self.resource_type,
                attributes=attrs,
                arch=self.client_info["arch"]
            )
            logging.info("Resource #%s created successfully.", res.id)
