import io
import re
import os
import sys
import cgi
import copy
import time
import errno
import random
import shutil
import signal
import logging
import tarfile
import datetime
import textwrap
import itertools as it
import compileall
import contextlib
import collections

import concurrent.futures

import six

from sandbox.common import fs as common_fs
from sandbox.common import urls as common_urls
from sandbox.common import enum as common_enum
from sandbox.common import rest as common_rest
from sandbox.common import config as common_config
from sandbox.common import errors as common_errors
from sandbox.common import encoding as common_encoding
from sandbox.common import patterns as common_patterns
from sandbox.common import platform as common_platform
from sandbox.common import telegram as common_telegram
from sandbox.common import itertools as common_it
from sandbox.common.types import task as ctt
from sandbox.common.types import misc as ctm
from sandbox.common.types import client as ctc
from sandbox.common.types import resource as ctr
from sandbox.common.types import notification as ctn

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects import resource_types
from sandbox.projects.common.ui import build_ui
from sandbox.projects.common.vcs import arc as arc_module
from sandbox.projects.common.arcadia import sdk as arc_sdk
from sandbox.projects.common.constants import constants as sdk_constants


from sandbox.projects.sandbox import resources as sb_resources
from sandbox.projects.sandbox.common import fs as sb_common_fs
from sandbox.projects.sandbox.common import get_binary_name

from sandbox.projects.sandbox.build_sandbox import ci_checks


# Users that can release stable tasks regardless of precommit checks
SUPER_RELEASERS = (
    "korum", "r-vetrov"
)

SKYNET_PYTHON_PATH = "/skynet/python/bin/python"


class Chats(common_enum.Enum):
    DEV = -1001075746847  # sandbox-dev
    STEP = -1001147285631  # Statistics Infrastructure
    RELEASES = -1001226139167  # sandbox-releases


class InfraConsts(object):
    BASE_URL = "https://infra-api.yandex-team.ru/v1"
    SERVICE = 50
    ENVIRONMENTS = {
        ctt.ReleaseStatus.STABLE: 75,
        ctt.ReleaseStatus.PRESTABLE: 140,
    }
    TYPE = "maintenance"
    SEVERITY = "minor"
    EVENT_DURATION = 30 * 60  # seconds
    SERVICE_DC_OUTAGE_ID = 154  # service to catch dc drills


class Vaults(object):
    INFRA = ("robot-sandbox", "infra-token")
    TRACKER = ("robot-sandbox", "startrek-token")
    ARCANUM = ("SANDBOX", "arcanum-token")


class BuildTarget(object):
    __slots__ = ("path", "platforms", "build_opts", "binary_name")

    def __init__(self, path, platforms, build_opts=None, binary_name=None):  # type: (str, list[str], dict, str) -> None
        self.path = path
        self.platforms = platforms
        self.build_opts = build_opts or {}
        self.binary_name = binary_name


# The first path is main python package, revision number will be embedded into its "__init__.py" file.
INC_PATHS = {
    "client": (
        "yasandbox",
        "web", "common", "agentr", "etc", "serviceq",
        "sandboxsdk", "sdk2",
        "client",
        "bin/client.py", "bin/executor.py", "bin/coredumper.py",
        "fileserver",
        "tasks",
        "executor",
        "taskbox",  # TODO: drop it SANDBOX-5884
    ),
    "agentr": ("agentr", "common", "etc", "web/__init__.py", "web/api"),
    "server": (
        "yasandbox", "services", "taskbox",
        "web", "common", "agentr", "scripts", "etc", "serviceq",
        "sandboxsdk", "sdk2", "serviceapi",
        "bin/server.py", "bin/shell.py",
        "deploy",
    ),
    "serviceapi": (
        "serviceapi", "common", "etc", "web",
        "yasandbox/__init__.py", "yasandbox/database/__init__.py", "yasandbox/database/mapping",
    ),
    "taskbox": (
        "taskbox", "common", "etc", "sdk2", "agentr", "web",
        "yasandbox/__init__.py", "yasandbox/database/__init__.py", "yasandbox/database/mapping",
    ),
    "serviceq": (
        "serviceq", "common", "etc",
        "yasandbox/__init__.py", "yasandbox/database/__init__.py", "yasandbox/database/mapping",
    ),
    "services": (
        "services", "common", "etc",
        "yasandbox/__init__.py", "yasandbox/database/__init__.py", "yasandbox/database/mapping",
        "yasandbox/database/clickhouse",
    ),
    "server-tests": (
        "devbox",
        "conftest.py", "pytest.ini",
        "bin/__init__.py",
        "tests",
    ),
    "proxy": ("proxy", "common", "web/__init__.py", "web/api", "etc"),
    "library": ("common",),
    "deploy": ("deploy",),
    "ui": (),
    "fileserver": ("fileserver", "common", "etc"),
    "preexecutor": (),
}


PACKAGE_TYPE = {
    "client": sb_resources.SandboxArchive,
    "preexecutor": sb_resources.SandboxArchive,
    "fileserver": sb_resources.SandboxArchive,
    "agentr": sb_resources.SandboxArchive,
    "server": sb_resources.SandboxArchive,
    "serviceapi": sb_resources.SandboxArchive,
    "taskbox": sb_resources.SandboxArchive,
    "serviceq": sb_resources.SandboxArchive,
    "services": sb_resources.SandboxArchive,
    "server-tests": sb_resources.SandboxArchive,
    "proxy": sb_resources.SandboxArchive,
    "library": sb_resources.SandboxArchive,
    "deploy": sb_resources.SandboxSamogonPlugin,
}

STARTREK_BASE_URL = "https://st-api.yandex-team.ru/v2"

TICKETS_RE = re.compile(r"([A-Z]+-\d+)")
REVIEW_RE = re.compile(r"REVIEW:\s*(\d+)")

STARTREK_URL = "https://st.yandex-team.ru"
ARCANUM_URL = "https://a.yandex-team.ru"


def paths_of_component(name):
    return INC_PATHS[name] + ("__init__.py", ".revision")


def safe_make_folder(path):
    try:
        common_fs.make_folder(path)
    except OSError as exc:
        if exc.errno == errno.EEXIST:
            pass


def prepare_commit_message(commit, issue_fetcher=None):
    message = commit["msg"]
    issues = TICKETS_RE.findall(message)
    reviews = REVIEW_RE.findall(message)
    if issue_fetcher is not None:
        issues.extend(it.chain.from_iterable(issue_fetcher[review] for review in reviews))

    def filter_line(line):
        line = line.strip().rstrip(".").lower()
        return not any((
            not line,
            line.endswith("quickfix"),
            line.startswith("review:"),
            line.startswith("[arc::pullid]"),
            line.startswith("issue:"),
            "NEED_CHECK" in line,
        ))

    return {
        "revision": commit["revision"],
        "author": commit["author"],
        "msg": ". ".join(filter(filter_line, message.splitlines())),
        "issues": sorted(set(issues)),
        "reviews": reviews,
    }


def sort_revisions(revisions):
    if isinstance(revisions, dict):
        revisions = revisions.values()
    return sorted(revisions, key=lambda x: x["revision"], reverse=True)


class IssueFetcher(object):
    __metaclass__ = common_patterns.SingletonMeta

    BASE_URL = "{}/api".format(ARCANUM_URL)

    def __init__(self, arcanum_token):
        self.api = common_rest.Client(self.BASE_URL, auth=arcanum_token)
        self.__cache = {}

    def __getitem__(self, review):
        if review not in self.__cache:
            try:
                review_data = self.api.review["review-request"][review].read()
                self.__cache[review] = [bug["name"] for bug in review_data["bugsClosed"]]
            except Exception as exc:
                logging.error("Failed to fetch issues attached to review r%s: %s", review, exc)
        return self.__cache[review]


class BuildTypeParameter(sdk2.parameters.String):
    @classmethod
    def cast(cls, value):
        if value not in ctt.ReleaseStatus:
            raise ValueError("Incorrect build type")
        return value


class BuildSandbox(sdk2.ServiceTask):
    """
    Task for building Sandbox package for release

    The input is the name of Sandbox's branch.
    The task tests it, on success creates tag and tarball(s).
    """

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL & ctc.Tag.INTEL_GOLD_6338
        disk_space = 16 << 10
        ram = 40 << 10
        cores = 16
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 3600  # 1 hour timeout for tests run
        release_to = ["sandbox-releases"]

        sandbox_branch_name = sdk2.parameters.String(
            "Url for arcadia or branch[@revision])", default="trunk", required=True
        )
        precompile = sdk2.parameters.Bool("Pre-compile python modules", default=True)
        include_symbols = sdk2.parameters.Bool("Include symbols from separate files to archives", default=True)
        force_release = sdk2.parameters.Bool("Ignore DC drills events on release", default=False, do_not_copy=True)

        with sdk2.parameters.Group("Patch") as base_build:
            arcadia_patch = sdk2.parameters.String(
                "Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). "
                "Doc: https://nda.ya.ru/3QTTV4",
                default="", multiline=True
            )
            patch_from_rb = sdk2.parameters.Bool("Apply patch from arcadia root [deprecated]", default=True)

        with sdk2.parameters.Group("Tests") as test_group:
            skip_tests = sdk2.parameters.Bool("Do not run any tests", do_not_copy=True, default=False)
            with skip_tests.value[False]:
                run_sq_perf_tests = sdk2.parameters.Bool("Run ServiceQ performance tests", default=False)
                parallel_tests_executing = sdk2.parameters.Bool("Run tests in parallel", default=True)
                stop_on_first_failure = sdk2.parameters.Bool("Stop tests on first failure", default=False)
                use_local_mongodb = sdk2.parameters.Bool("Use local MongoDB instance", default=True)

        with sdk2.parameters.Group("Sandbox components") as components_group:
            create_client_resource = sdk2.parameters.Bool("Client", do_not_copy=True)
            build_fileserver = sdk2.parameters.Bool("Fileserver", do_not_copy=True)
            create_agentr_resource = sdk2.parameters.Bool("AgentR", do_not_copy=True)
            create_server_resource = sdk2.parameters.Bool("Server", do_not_copy=True)
            create_serviceapi_resource = sdk2.parameters.Bool("ServiceApi", do_not_copy=True)
            create_taskbox_resource = sdk2.parameters.Bool("Taskbox", do_not_copy=True)
            create_serviceq_resource = sdk2.parameters.Bool("ServiceQ", do_not_copy=True)
            create_services_resource = sdk2.parameters.Bool("Services", do_not_copy=True)
            create_proxy_resource = sdk2.parameters.Bool("Proxy", do_not_copy=True)
            create_library_resource = sdk2.parameters.Bool("REST client library", do_not_copy=True)
            create_samogon_resource = sdk2.parameters.Bool("Samogon plugin")

        with create_samogon_resource.value[True]:
            venv_platforms = sdk2.parameters.List(
                "Platforms for which to get actual virtual environments",
                default=[
                    "linux_ubuntu_12.04_precise",
                    "linux_ubuntu_14.04_trusty",
                    "linux_ubuntu_16.04_xenial",
                    "linux_ubuntu_18.04_bionic",
                    "linux_ubuntu_20.04_focal",
                    "osx_10.13_high_sierra",
                    "osx_10.14_mojave",
                    "osx_10.15_catalina",
                    "osx_10.16_big_sur",
                    "osx_12_monterey",
                    "cygwin_6.3",
                ],
            )

        with BuildTypeParameter("Build type", multiline=True) as build_type:
            build_type.values[""] = build_type.Value(value="not selected", default=True)
            for name, value in common_it.chain(((x, x) for x in ctt.ReleaseStatus)):
                build_type.values[name] = build_type.Value(value=value)

    class Context(sdk2.Context):
        sandbox_version = None
        pids_to_alarm = []

    def create_resource(self, resource_type, description, path, arch=None, attrs=None):
        if attrs is None:
            attrs = {}
        for_parent = self is not self.current
        if arch is None:
            arch = ctm.OSFamily.ANY if resource_type.any_arch else common_config.Registry().this.system.family
        assert arch in ctm.OSFamily, "arch must be one of {!r}".format(ctm.OSFamily)
        if for_parent:
            attrs["from_task"] = self.current.id
        data = self.current.agentr.resource_register(
            path, str(resource_type), description, arch, attrs,
            share=resource_type.share, service=issubclass(resource_type, sdk2.ServiceResource),
            for_parent=for_parent,
        )
        return resource_type.restore(data)

    def get_revisions(self, fetch_checks=False):
        revisions = copy.deepcopy(self.Context.revisions) or []
        if not (revisions and fetch_checks):
            return revisions
        checks = ci_checks.recent_checks([r["revision"] for r in revisions[:30]])
        for rev in revisions:
            check = checks.get(rev["revision"])
            if check:
                rev["check_task"] = check.task
                rev["check_status"] = check.status

        return revisions

    @sdk2.header()
    def header(self):
        if self.status not in (ctt.Status.SUCCESS, ctt.Status.NOT_RELEASED, ctt.Status.RELEASED):
            return

        tplt = (
            "<span style='font-size:15px'>"
            "<span style='color:{color};font-weight:bold;display:block'>{title}</span>{text}"
            "</span>"
        )

        if self.status == ctt.Status.RELEASED:
            check = self.Context.most_recent_check
            if not check:
                return
            check = ci_checks.Check(**check)
            return tplt.format(color="green", title="Released", text="r{}:{}".format(check.revision, check.status))

        start_event = self.Context.start_dc_drills_time
        end_event = self.Context.end_dc_drills_time
        check_dc = self.check_for_dc_drills(start_event, end_event)
        if not check_dc.can_release:
            return tplt.format(color="red", title="Cannot release", text=check_dc.description)

        try:
            check = self.validate_release()
            if check is None:
                return
            text = "r{}:{}".format(check.revision, check.status)
            return tplt.format(color="green", title="Ready for release", text=text)
        except common_errors.ReleaseError as exc:
            return tplt.format(color="red", title="Cannot release", text=exc.message)

    @sdk2.footer()
    def footer(self):
        fetch_checks = not bool(self.Context.checks_cached)
        revisions = self.get_revisions(fetch_checks=fetch_checks)

        if not revisions:
            return None

        rows = []

        for rev in revisions:
            try:
                message = rev["msg"].splitlines()[0].strip()
            except IndexError:
                message = ""

            row = collections.OrderedDict((
                ("Revision", '<a href="{0}/arc/commit/{1}">{1}</a>'.format(ARCANUM_URL, rev["revision"])),
                ("Author", '<a href="https://staff.yandex-team.ru/{0}">{0}@</a>'.format(rev["author"])),
                ("Message", message),
                ("Tickets", ", ".join(
                    '<a href="{0}/{1}">{1}</a>'.format(STARTREK_URL, t) for t in rev["issues"]
                )),
                ("Reviews", ", ".join(
                    '<a href="{0}/review/{1}/details">{1}</a>'.format(ARCANUM_URL, r) for r in rev["reviews"]
                )),
                ("Tests", ""),
            ))

            if "check_task" in rev:
                row["Tests"] = (
                    '<a href="https://sandbox.yandex-team.ru/task/{task}">'
                    '<span class="status status__padded status_{status_lower}">{status}</span>'
                    '</a>'
                ).format(
                    task=rev["check_task"],
                    status=rev["check_status"],
                    status_lower=rev["check_status"].lower(),
                )

            rows.append(row)

        return {"<h3>Release Information</h3>": rows}

    @property
    def release_template(self):
        subject = "Release Sandbox tag {}".format(self.Context.sandbox_version)
        changelogs = self.Context.sandbox_changelogs or {}
        changelogs = collections.OrderedDict(sorted(changelogs.iteritems(), key=lambda item: item[0]))
        release_tag = self.Parameters.build_type
        release_str = (
            'Changelog from previous "{}" release tag:'.format(release_tag)
            if release_tag else
            "Changelog from previous built resource:"
        )

        buf = io.StringIO()
        for name, chlog in changelogs.items():
            if chlog is None:
                continue

            buf.write(u"{}@{}:{} (resource #{}, task #{})\n".format(
                name.upper(),
                chlog.get("rev_from"),
                chlog.get("rev_to"),
                chlog.get("base_resource_id"),
                chlog.get("base_resource_task_id")
            ))

            for revision in sort_revisions(chlog.get("revisions") or []):
                buf.write(u"  -- ")
                buf.write(common_encoding.force_unicode_safe(revision["msg"]))
                buf.write(u"\n")
                if revision["issues"]:
                    for issue in revision["issues"]:
                        buf.write(u"    -- {}/{}\n".format(STARTREK_URL, issue))

            buf.write(u"\n")

        return sdk2.ReleaseTemplate(
            [],
            subject,
            u"{}\n\n{}".format(release_str, buf.getvalue()),
            [release_tag, ctt.ReleaseStatus.CANCELLED]
        )

    @staticmethod
    def create_changelog(code_info, component, archive_type, release_tag, svn_path_prefix, issue_fetcher):
        attrs = {
            "type": archive_type
        }

        if release_tag:
            attrs["released"] = release_tag

        prev_resource = sdk2.Resource.find(
            PACKAGE_TYPE[component],
            status=ctr.State.READY,
            limit=1,
            attrs=attrs
        ).first()

        if prev_resource is None:
            logging.warning("Changelog: no previous resource with attr.type=%s.", archive_type)
            return

        if not hasattr(prev_resource, "version"):
            logging.warning("Changelog: can't parse, unexpected resource attributes: %r.", prev_resource.attributes)
            return

        prev_rev = prev_resource.version
        if prev_rev.startswith("r"):
            prev_rev = prev_rev[1:]
        prev_rev = int(prev_rev) + 1

        arch_paths = INC_PATHS[component]
        if svn_path_prefix:
            arch_paths = ["/".join([svn_path_prefix, path]) for path in arch_paths]
        logging.debug("Archive paths: %s", arch_paths)

        revisions = {}
        for line in sdk2.svn.Arcadia.log(code_info.url, prev_rev, code_info.rev):
            logging.debug("Svn log record: %s", line)
            for action, path in line["paths"]:
                if any(path.startswith(arch_p) for arch_p in arch_paths):
                    revisions[line["revision"]] = prepare_commit_message(line, issue_fetcher)
                    break

        return {
            "rev_from": prev_rev,
            "rev_to": code_info.rev,
            "base_resource_id": prev_resource.id,
            "base_resource_task_id": prev_resource.task_id,
            "revisions": revisions,
        }

    def on_save(self):
        has_telegram = any(
            notification.transport == ctn.Transport.TELEGRAM
            for notification in self.Parameters.notifications
        )
        if self.Parameters.create_samogon_resource and not has_telegram:
            self.Parameters.notifications += [
                sdk2.Notification(
                    [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK],
                    [self.author],
                    ctn.Transport.TELEGRAM
                ),
            ]

    def on_enqueue(self):
        if self.Parameters.use_local_mongodb:
            self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 1024, None)

    def on_before_timeout(self, seconds):
        super(BuildSandbox, self).on_before_timeout(seconds)
        if seconds == min(self.timeout_checkpoints()):
            for pid in self.Context.pids_to_alarm:
                logging.warning("Send SIGALRM to process #%s", pid)
                os.kill(int(pid), signal.SIGALRM)

    def create_sandbox_archive(self, sandbox_path, sandbox_version, archive_type):
        """
            Create archive file with Sandbox code

            :param sandbox_path: path to directory with Sandbox code
            :param sandbox_version: version of Sandbox - tag name
            :param archive_type:
            :return: path to archive file
        """
        if not os.path.exists(sandbox_path):
            raise common_errors.TaskFailure("Folder {} does not exist.".format(sandbox_path))

        logging.info("Creating Sandbox %s archive from path %s.", archive_type, sandbox_path)
        archive_path = str(self.path("sandbox-{}.{}.tar.gz".format(archive_type, sandbox_version.replace("/", "_"))))

        component = archive_type.split("-", 1)[0]  # strip platform tag
        inc_paths = [os.path.join(sandbox_path, p) for p in paths_of_component(component)]

        def exclude_func(path):
            return not any(
                path == p or path.startswith(p + os.sep) or p.startswith(path + os.sep)
                for p in inc_paths
            )

        with contextlib.closing(tarfile.open(archive_path, "w:gz", dereference=True)) as tar_file:
            for file_name in os.listdir(sandbox_path):
                path_in_archive = file_name
                if file_name != ".revision":
                    path_in_archive = os.path.join("sandbox", file_name)
                tar_file.add(
                    os.path.join(sandbox_path, file_name),
                    arcname=path_in_archive,
                    exclude=exclude_func if inc_paths else None,
                )

            if archive_type.startswith("serviceapi-"):
                binary_path = str(self.path(archive_type, "sandbox", "serviceapi", "serviceapi"))
                tar_file.add(binary_path, arcname="serviceapi")

            if archive_type.startswith("taskbox-"):
                binary_path = str(self.path(archive_type, "sandbox", "taskbox", "dispatcher", "taskbox"))
                tar_file.add(binary_path, arcname="taskbox")

            if archive_type.startswith("fileserver-"):
                binary_path = str(self.path(archive_type, "sandbox", "fileserver", "fileserver"))
                tar_file.add(binary_path, arcname="sandbox/fileserver/fileserver")

            if archive_type.startswith("services-"):
                binary_path = str(self.path(archive_type, "sandbox", "services", "sandbox-services"))
                tar_file.add(binary_path, arcname="sandbox-services")

            if archive_type.startswith("preexecutor"):
                if archive_type == "preexecutor-windows":
                    binary_path = str(self.path(archive_type, "sandbox", "executor", "preexecutor", "preexecutor.exe"))
                    tar_file.add(binary_path, arcname="preexecutor.exe")
                else:
                    binary_path = str(self.path(archive_type, "sandbox", "executor", "preexecutor", "preexecutor"))
                    tar_file.add(binary_path, arcname="preexecutor")

            if archive_type.startswith("client-"):
                binary_path = str(self.path(archive_type, "sandbox", "client", "bin", "client"))
                tar_file.add(binary_path, arcname="client")

            if archive_type.startswith("serviceq-"):
                binary_path = str(self.path(archive_type, "sandbox", "serviceq", "bin", "serviceq"))
                tar_file.add(binary_path, arcname="serviceq")

            if archive_type.startswith("proxy-"):
                binary_path = str(self.path(archive_type, "sandbox", "proxy", "proxy"))
                tar_file.add(binary_path, arcname="proxy")

            if archive_type.startswith("agentr-"):
                binary_path = str(self.path(archive_type, "sandbox", "agentr", "bin", "agentr"))
                tar_file.add(binary_path, arcname="agentr")

                platform = archive_type[len("agentr-"):]
                binary_path = str(self.path("-".join(("bctl", platform)), "sandbox", "agentr", "bin", "bctl", "bctl"))
                tar_file.add(binary_path, arcname="bctl")

                binary_path = str(
                    self.path("-".join(("reshare", platform)), "sandbox", "agentr", "bin", "reshare", "reshare")
                )
                tar_file.add(binary_path, arcname="reshare")

                tar_file.add(str(self.path("infra")), arcname="infra")

                # adding symbols for macOS; for linux it is embedded into the binary
                if self.Parameters.include_symbols and "darwin" in platform:
                    symbols_path = str(self.path(archive_type, "sandbox", "agentr", "bin", "agentr.dSYM"))
                    logging.info("Including symbols '%s' for platform: %s", symbols_path, platform)
                    tar_file.add(symbols_path, arcname="agentr.dSYM")

        return archive_path

    def _run_tests(self, sandbox_svn_dir, sandbox_tasks_dir, ext_params=None, run_in_parallel=True):
        env = dict(os.environ)
        for k in ("PWD", "OLDPWD", "SANDBOX_CONFIG"):
            env.pop(k, None)
        env["PYTHONPATH"] = ":".join(("/skynet", sandbox_tasks_dir))
        env["LC_ALL"] = ""

        def run_process(parallel, clear_cache):
            pytest_args = [
                ".", "-v", "--timeout=180", "--durations=0", "--tasks={}".format(sandbox_tasks_dir), "--last-failed"
            ]
            if clear_cache:
                pytest_args.append("--cache-clear")
            if not parallel:
                pytest_args.append("-n0")
            pytest_args.extend(ext_params or [])
            with sdk2.helpers.ProcessLog(self, logger="tests") as pl:
                process = sp.Popen(
                   [sys.executable, "-c", "import sys, pytest; sys.exit(pytest.main({!r}))".format(pytest_args)],
                   stdout=pl.stdout,
                   stderr=sp.STDOUT,
                   env=env
                )
                self.Context.pids_to_alarm.append(process.pid)
                process.wait()
                self.Context.pids_to_alarm = [_ for _ in self.Context.pids_to_alarm if _ != process.pid]
                if process.returncode:
                    raise common_errors.SubprocessError(
                        "tests failed (exit code: {}), see logs for details".format(process.returncode)
                    )

        parallel_tests_executing = run_in_parallel and self.Parameters.parallel_tests_executing
        with common_fs.WorkDir(sandbox_svn_dir):
            try:
                run_process(parallel_tests_executing, True)
            except common_errors.SubprocessError as ex:
                if parallel_tests_executing:
                    self.set_info("Parallel tests failed. Try one more time")
                    try:
                        run_process(parallel_tests_executing, False)
                    except common_errors.SubprocessError as ex:
                        raise common_errors.TaskFailure(ex)
                else:
                    raise common_errors.TaskFailure(ex)

    def _prepare_arc_url(self):  # type: () -> (str, str)
        # CI format
        if self.Parameters.sandbox_branch_name.startswith("arcadia-arc:"):
            return str(self.Parameters.sandbox_branch_name), "trunk"
        # Backward-compatible custom format
        branch, rev = (self.Parameters.sandbox_branch_name.split("@", 1) + [None])[:2]
        if branch == "trunk":
            return sdk2.svn.Arcadia.trunk_url(revision=rev), branch
        return sdk2.svn.Arcadia.branch_url(branch, revision=rev), branch

    def build_binaries_chunk(self, arc_path, patch, platform, builds):
        # type: (str, str, str, list[BuildTarget]) -> None
        results_dir = str(self.path("results_dir"))
        build_options = {
            "target_platform": platform,
            "clear_build": False,
            "results_dir": results_dir,
            "patch": patch
        }
        for build in builds:
            build_options.update(build.build_opts)
        targets = [build.path for build in builds]
        logging.info("Build targets: %s\nBuild opts: %s\n", targets, build_options)
        build_return_code = arc_sdk.do_build(
            build_system=sdk_constants.SEMI_DISTBUILD_BUILD_SYSTEM,
            source_root=arc_path,
            targets=targets,
            **build_options
        )
        binaries = [
            get_binary_name(arc_path, target, build_options["results_dir"])
            for target in targets
        ]
        logging.info("Binaries: %s", binaries)
        if build_return_code:
            raise common_errors.SubprocessError("Build failed with exit code {}".format(build_return_code))
        for idx, binary in enumerate(binaries):
            dst_dir = str(self.path(
                "{}-{}".format(builds[idx].binary_name or os.path.basename(builds[idx].path), platform),
                targets[idx]
            ))
            os.makedirs(dst_dir)
            dst = os.path.join(dst_dir, os.path.basename(binary))
            logging.info("Move binary %s to %s", binary, dst)
            shutil.move(binary, dst)

            symbols_basename = "{}.dSYM".format(os.path.basename(binary))
            src = os.path.join(os.path.dirname(binary), symbols_basename)
            if os.path.exists(src):
                logging.info("Move symbols %s to %s", src, dst_dir)
                shutil.move(src, dst_dir)
        shutil.rmtree(results_dir)

    def build_binaries(self, arc_path, patch, builds):  # type: (str, str, list[BuildTarget]) -> None
        build_by_platforms = collections.defaultdict(list)
        for build in builds:
            for platform in build.platforms:
                build_by_platforms[platform].append(build)
        for platform, platform_builds in build_by_platforms.items():
            simple = []
            for build in platform_builds:
                if build.build_opts:
                    self.build_binaries_chunk(arc_path, patch, platform, [build])
                else:
                    simple.append(build)
            if simple:
                self.build_binaries_chunk(arc_path, patch, platform, simple)

    def _build_component_binaries(self, arc_path, patch):  # type: (str, str) -> None
        builds = []
        if self.Parameters.build_fileserver:
            builds.append(BuildTarget("sandbox/fileserver", ["linux", "darwin", "default-darwin-arm64"]))
        if self.Parameters.create_serviceapi_resource:
            builds.append(BuildTarget("sandbox/serviceapi", ["linux"]))
        if self.Parameters.create_taskbox_resource:
            builds.append(BuildTarget("sandbox/taskbox/dispatcher", ["linux"], binary_name="taskbox"))
        if self.Parameters.create_services_resource:
            builds.append(BuildTarget("sandbox/services", ["linux"]))
        if self.Parameters.create_client_resource:
            builds.append(BuildTarget(
                "sandbox/client/bin", ["darwin", "linux", "default-darwin-arm64"], binary_name="client"
            ))
            builds.append(BuildTarget(
                "sandbox/executor/preexecutor", ["windows", "darwin", "linux", "default-darwin-arm64"]
            ))
        if self.Parameters.create_agentr_resource:
            builds.append(BuildTarget(
                "sandbox/agentr/bin", ["darwin", "linux", "default-darwin-arm64"], binary_name="agentr"
            ))
            builds.append(BuildTarget(
                "sandbox/agentr/bin/bctl", ["darwin", "linux", "default-darwin-arm64"], binary_name="bctl"
            ))
            builds.append(BuildTarget(
                "sandbox/agentr/bin/reshare", ["darwin", "linux", "default-darwin-arm64"], binary_name="reshare"
            ))
        if self.Parameters.create_serviceq_resource:
            builds.append(
                BuildTarget(
                    "sandbox/serviceq/bin", ["linux"],
                    build_opts={"lto": True, "clear_build": True},
                    binary_name="serviceq"
                )
            )
        if self.Parameters.create_proxy_resource:
            builds.append(BuildTarget("sandbox/proxy", ["linux"]))
        if builds:
            os.mkdir(str(self.path("arc")))
            self.build_binaries(arc_path, patch, builds)

    def setup_local_mongodb(self):  # type: () -> list[str]
        if not self.Parameters.use_local_mongodb:
            return []
        logging.info("Downloading and starting local instance of MongoDB")
        resource = resource_types.MONGO_PACKAGE.find(attrs=dict(version="3.4.10")).limit(1).first()
        logging.debug("Found MongoDB package resource #%r", resource.id)
        out = sp.check_output(["tar", "-zxvf", str(sdk2.ResourceData(resource).path)])
        mongodb_dir = out.split("\n", 1)[0].split("/", 1)[0]
        with open(mongodb_dir + "/mongod.conf", "w") as fh:
            fh.write(textwrap.dedent(u"""
                storage:
                  dbPath: {db_path}
                  journal:
                    enabled: false
                systemLog:
                  destination: file
                  logAppend: true
                  path: {log_path}
                net:
                  port: 27017
                  bindIp: 127.0.0.1
            """).format(db_path=self.ramdrive.path, log_path=self.log_path("mongod.log")))
        sp.Popen(["bin/mongod", "--config", "mongod.conf"], cwd=mongodb_dir)
        # The process will be killed after task finish by the task executor
        return ["--mongo-uri=mongodb://127.0.0.1"]

    def _precompile_sandbox_sources(self, arc_path, revision):  # type: (str, str) -> None
        sandbox_dir = sdk2.path.Path(arc_path) / "sandbox"
        logging.info("Compile code and embed revision %r metadata", revision)
        with (sandbox_dir / ".revision").open("w") as revision_file:
            revision_file.write(six.text_type(revision))
        for ips in INC_PATHS.values():
            if ips:
                sb_common_fs.embed_revision(sandbox_dir / ips[0] / "__init__.py", str(revision))
        compileall.compile_dir(str(sandbox_dir), force=True)
        # remove all *.pyc files from layout
        map(sdk2.path.Path.unlink, (sandbox_dir / "deploy" / "layout").glob("**/*.pyc"))

    def _run_sandbox_tests(self, arc_path, arc_url):  # type: (str, str) -> None
        sandbox_dir, runtime_dir, tasks_dir = self._prepare_tests_environment(arc_path, arc_url)
        sdk2.ResourceData(
            sb_resources.SandboxTestResults(self, self.Parameters.description, runtime_dir)
        )
        local_mongo_args = self.setup_local_mongodb()
        runtime_args = ["--runtime", runtime_dir]
        # Flake tests
        extra_args = [
            "--flakes", "-m", "flakes",
            "--junit-xml", str(self.log_path("test_flakes_results.xml")),
        ] + runtime_args
        if self.Parameters.stop_on_first_failure:
            extra_args.append("--exitfirst")
        self.set_info("Flakes tests are running")
        self._run_tests(sandbox_dir, tasks_dir, ext_params=extra_args, run_in_parallel=False)
        logging.info("Flakes tests completed")
        # Common tests
        extra_args = [
            "-m",
            "not sdk_svn and not serviceq_replication and not serviceq_perf and not test_tasks and not flakes",
            "--junit-xml", str(self.log_path("test_results.xml")),
        ] + local_mongo_args + runtime_args
        if self.Parameters.stop_on_first_failure:
            extra_args.append("--exitfirst")
        self.set_info("Main tests are running")
        self._run_tests(sandbox_dir, tasks_dir, ext_params=extra_args)
        logging.info("Main tests completed")
        # ServiceQ tests
        extra_args = [
            "-m", "serviceq_replication and not flakes",
            "--junit-xml", str(self.log_path("serviceq_replication_tests.xml")),
        ] + local_mongo_args + runtime_args
        self.set_info("ServiceQ replication tests are running")
        self._run_tests(sandbox_dir, tasks_dir, ext_params=extra_args, run_in_parallel=False)
        logging.info("ServiceQ replication tests completed")
        # Tasks tests
        extra_args = [
            "-m", "test_tasks and not flakes",
            "--junit-xml", str(self.log_path("test_tasks_results.xml")),
        ] + local_mongo_args + runtime_args
        self.set_info("Long tests are running")
        self._run_tests(sandbox_dir, tasks_dir, ext_params=extra_args)
        logging.info("Long tests completed")
        # Optional ServiceQ performance tests
        if self.Parameters.run_sq_perf_tests:
            extra_args = [
                "-m", "serviceq_perf and not flakes",
                "--junit-xml", str(self.log_path("serviceq_perf_tests.xml")),
            ] + local_mongo_args + runtime_args
            self.set_info("ServiceQ performance tests are running")
            self._run_tests(sandbox_dir, tasks_dir, ext_params=extra_args)
            logging.info("ServiceQ performance tests completed")
        self.set_info("All tests completed")

    def _prepare_tests_environment(self, arc_path, arc_url):  # type: (str, str) -> (str, str, str)
        arcadia_path = sdk2.path.Path(arc_path)
        tasks_resource = self._get_sandbox_tasks_resource(arc_url)
        # Tests dirs
        tests_dir = self.path("sandbox_tests")
        tasks_dir = tests_dir / "sandbox_tasks"
        runtime_dir = tests_dir / "runtime_data"
        # Target sources
        sandbox_svn_dir = arcadia_path / "sandbox"
        taskboxer_dir = str(sandbox_svn_dir / "projects/sandbox/taskboxer/isolated")
        targets = [
            str(sandbox_svn_dir / name)
            for name in (
                "fileserver", "serviceapi", "taskbox", "agentr/bin", "scripts/py3_sources",
                "executor/preexecutor"
            )
        ] + [taskboxer_dir]
        # Build binaries for tests
        with sdk2.helpers.ProcessLog(self, logger="ya_make_binaries") as pl:
            sp.check_call(
                # don't use build=release, tests want regular build
                [SKYNET_PYTHON_PATH, str(arcadia_path / "ya"), "make", "--thin"] + targets,
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            )
        # Isolated taskboxer dir is not included to tasks archive that is based on code reachable
        #   from `sandbox/projects`. So we have to put taskboxer to tasks dir for proper testing.
        safe_taskboxer_dir = str(self.path("tmp/taskboxer"))
        shutil.copytree(taskboxer_dir, safe_taskboxer_dir)
        # Some tests (for example, devbox/tests/ctl_impl.TestCTL.test__setup) require tested tasks code,
        #   hence we need to replace checked out sandbox/projects with an approved archive contents
        tasks_resource_path = sdk2.ResourceData(sdk2.Resource[tasks_resource.id]).path
        shutil.rmtree(str(sandbox_svn_dir / "projects"))
        for target in (tasks_dir, sandbox_svn_dir):
            tarfile.open(str(tasks_resource_path), "r:gz").extractall(str(target))
        if not os.path.exists(taskboxer_dir):
            shutil.copytree(safe_taskboxer_dir, taskboxer_dir)
        runtime_dir.mkdir(parents=True, exist_ok=True)
        return str(sandbox_svn_dir), str(runtime_dir), str(tasks_dir)

    def _get_sandbox_tasks_resource(self, arc_url):  # type: (str) -> sb_resources.SandboxTasksArchive
        logging.info("Get the last resource with tasks source code")
        tasks_resource = sb_resources.SandboxTasksArchive.find(
            owner=common_config.Registry().common.service_group,
            state=ctr.State.READY,
            attrs=dict(
                auto_deploy=True
            )
        ).order(-sdk2.Resource.id).first()
        if not tasks_resource:
            subtask = self.server.task(
                children=True,
                type="BUILD_SANDBOX_TASKS",
                owner=self.owner,
                custom_fields=[{"name": "tasks_svn_url", "value": arc_url}, {"name": "auto_deploy", "value": True}],
            )
            self.server.batch.tasks.start.update({"id": [subtask["id"]]})
            raise sdk2.WaitTask([subtask["id"]], [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK])
        logging.info("Tasks archive resource: #%d", tasks_resource.id)
        return tasks_resource

    def __create_sandbox_resource(self, code_info, archive_type, components):
        if archive_type == "deploy":
            return self.create_samogon_resource(code_info, components)
        sandbox_archive = self.create_sandbox_archive(code_info.path, code_info.version, archive_type)
        resource_description = "Sandbox {}: {}".format(archive_type, code_info.version)
        sdk2.ResourceData(self.create_resource(
            sb_resources.SandboxArchive,
            resource_description,
            sandbox_archive,
            attrs=dict(type=archive_type, version=code_info.rev)
        ))
        return sandbox_archive

    def create_samogon_resource(self, code_info, components):  # type: (build_ui.CodeInfoSvn, dict[str,str]) -> str
        logging.info("Creating plugin for Samogon...")
        package_path = common_fs.make_folder(str(self.path("package")))

        plugin_path = os.path.join(package_path, "plugin")
        shutil.copytree(os.path.join(code_info.path, "deploy"), plugin_path)

        crossplatform_path = common_fs.make_folder(os.path.join(package_path, "crossplatform"))
        release_tag = self.Parameters.build_type

        # packages with Sandbox core code
        pkgs = {
            "server": None,
            "serviceq": None,
            "client-linux": "linux",
            "client-darwin": "osx",
            "client-default-darwin-arm64": "osx",
            "agentr-linux": "linux",
            "agentr-darwin": "osx",
            "agentr-default-darwin-arm64": "osx",
            "preexecutor-windows": None,
            "preexecutor-linux": "linux",
            "preexecutor-darwin": "osx",
            "preexecutor-default-darwin-arm64": "osx",
            "serviceapi-linux": "linux",
            "taskbox-linux": "linux",
            "services-linux": "linux",
            "fileserver-linux": "linux",
            "fileserver-darwin": "osx",
            "fileserver-default-darwin-arm64": "osx",
            "serviceq-linux": "linux",
            "proxy-linux": "linux",
        }

        def fetch_package(item):
            pkg_type, pkg_platform = item
            pkg_path = components.get(pkg_type)
            if not pkg_path:
                res = sb_resources.SandboxArchive.find(
                    state=ctr.State.READY,
                    attrs=dict(released=release_tag, type=pkg_type)
                ).order(-sdk2.Task.id).first()
                if res is None:
                    raise common_errors.TaskFailure(
                        "Cannot find resource {!r} of type {!r} with release tag {!r}".format(
                            sb_resources.SandboxArchive.name, pkg_type, release_tag
                        )
                    )
                pkg_path = str(sdk2.ResourceData(res).path)

            target_path = os.path.join(package_path, pkg_platform) if pkg_platform else crossplatform_path
            if pkg_platform:
                safe_make_folder(target_path)
            # Deploy preexecutor for windows and for linux to linux platforms
            if pkg_type == "preexecutor-windows" or pkg_type.endswith("default-darwin-arm64"):
                shutil.copyfile(pkg_path, os.path.join(target_path, pkg_type + ".tgz"))
            else:
                shutil.copyfile(pkg_path, os.path.join(target_path, pkg_type.split("-")[0] + ".tgz"))

        with concurrent.futures.ThreadPoolExecutor(max_workers=len(pkgs.items())) as pool:
            list(pool.map(fetch_package, pkgs.items()))

        def fetch_environment_package(platform):
            if not platform:
                return

            arch_path = os.path.join(package_path, common_platform.get_arch_from_platform(platform))
            safe_make_folder(arch_path)

            res = sb_resources.SandboxDeps.find(
                state=ctr.State.READY,
                attrs=dict(released=release_tag, platform=platform)
            ).order(-sdk2.Task.id).first()
            if res is None:
                raise common_errors.TaskFailure(
                    "Cannot find resource {!r} for platform {!r} with release tag {!r}".format(
                        sb_resources.SandboxDeps.name, platform, release_tag
                    )
                )
            pkg_path = str(sdk2.ResourceData(res).path)
            shutil.copyfile(pkg_path, os.path.join(arch_path, "venv_{}.tgz".format(platform)))

        # packages with Sandbox virtual environments
        with concurrent.futures.ThreadPoolExecutor(max_workers=len(self.Parameters.venv_platforms)) as pool:
            list(pool.map(fetch_environment_package, self.Parameters.venv_platforms))

        # package with Statinfra EventProcessor
        res = sb_resources.EventProcessor.find(
            state=ctr.State.READY,
            attrs=dict(released=release_tag)
        ).order(-sdk2.Task.id).first()
        if res is None:
            raise common_errors.TaskFailure(
                "Cannot find resource {!r} with release tag {!r}".format(
                    sb_resources.EventProcessor.name, release_tag
                )
            )
        pkg_path = str(sdk2.ResourceData(res).path)
        shutil.copyfile(
            os.path.join(pkg_path, "step.tgz"),
            os.path.join(crossplatform_path, "step.tgz")
        )
        shutil.copyfile(
            os.path.join(pkg_path, "venv_step.tgz"),
            os.path.join(package_path, "linux", "venv_step.tgz")
        )

        # package with REM to StEP converter binary
        settings = common_config.Registry()
        res = sb_resources.RemToStepConverter.find(
            state=ctr.State.READY,
            owner=settings.common.service_group,
            attrs=dict(released=release_tag)
        ).order(-sdk2.Task.id).first()
        if res is None:
            raise common_errors.TaskFailure(
                "Cannot find resource {!r} with release tag {!r}".format(
                    sb_resources.RemToStepConverter.name, release_tag
                )
            )
        pkg_path = str(sdk2.ResourceData(res).path)
        arch_path = os.path.join(package_path, "linux", "linux_rem_to_step.tgz")
        with contextlib.closing(tarfile.open(arch_path, "w:gz", dereference=True)) as tar_file:
            tar_file.add(pkg_path, arcname=os.path.basename(pkg_path))

        package_tgz_path = os.path.basename(package_path) + ".tgz"
        with tarfile.open(package_tgz_path, "w:gz") as fh:
            fh.add(package_path, arcname=os.path.basename(package_path))
        logging.debug("Successfully created %r archive from %r path", package_tgz_path, package_path)
        self.create_resource(
            sb_resources.SandboxSamogonPlugin,
            "Sandbox package with samogon plugin",
            package_tgz_path,
            attrs=dict(version=code_info.rev, type="deploy")
        )
        return package_tgz_path

    @common_patterns.singleton_property
    def telegram_bot(self):
        bot_token = sdk2.yav.Secret("sec-01g56fb670bc9azcrk1wfq8s2c").data()["token"]  # yasandbot-telegram-token
        return common_telegram.TelegramBot(bot_token)

    def on_prepare(self):
        if self.Parameters.build_type not in (ctt.ReleaseStatus.PRESTABLE, ctt.ReleaseStatus.STABLE):
            return

        components_list = [
            getattr(type(self).Parameters, param_name).description
            for param_name in type(self).Parameters.components_group.names
            if getattr(self.Parameters, param_name)
        ]
        if components_list and self.author in sb_resources.SandboxArchive.releasers:
            text = '{}@ is building a <strong>{}</strong> release (task <a href="{}">{}</a>) with {}'.format(
                self._releaser(),
                self.Parameters.build_type.upper(),
                common_urls.get_task_link(self.id),
                self.id,
                ", ".join(components_list),
            )
            try:
                self.telegram_bot.send_message(Chats.DEV, text, common_telegram.ParseMode.HTML)
            except Exception:
                logging.exception("Failed to send pre-release notification to Telegram")

    def _init_components(self):
        components = collections.OrderedDict()
        if self.Parameters.create_client_resource:
            components["client-linux"] = None
            components["client-darwin"] = None
            components["client-default-darwin-arm64"] = None
            components["preexecutor-windows"] = None
            components["preexecutor-darwin"] = None
            components["preexecutor-default-darwin-arm64"] = None
            components["preexecutor-linux"] = None
        if self.Parameters.build_fileserver:
            components["fileserver-linux"] = None
            components["fileserver-darwin"] = None
            components["fileserver-default-darwin-arm64"] = None
        if self.Parameters.create_agentr_resource:
            components["agentr-linux"] = None
            components["agentr-darwin"] = None
            components["agentr-default-darwin-arm64"] = None
        if self.Parameters.create_server_resource:
            components["server"] = None
            components["server-tests"] = None
        if self.Parameters.create_serviceapi_resource:
            components["serviceapi-linux"] = None
        if self.Parameters.create_taskbox_resource:
            components["taskbox-linux"] = None
        if self.Parameters.create_services_resource:
            components["services-linux"] = None
        if self.Parameters.create_serviceq_resource:
            components["serviceq-linux"] = None
        if self.Parameters.create_proxy_resource:
            components["proxy-linux"] = None
        if self.Parameters.create_samogon_resource:
            components["deploy"] = None
        return components

    def on_execute(self):
        components = self._init_components()
        if self.Parameters.create_library_resource and self.Parameters.build_type == ctt.ReleaseStatus.STABLE:
            self._run_common_package_build()
        arc_url, branch = self._prepare_arc_url()
        with arc_sdk.mount_arc_path(arc_url) as arc_path:
            revision, is_latest_commit = self.get_svn_revision(arc_path)
            logging.info(
                "Build sandbox from %s, revision %s%s",
                arc_url, revision, " (with patch)" if self.Parameters.arcadia_patch or not is_latest_commit else ""
            )
            patch = self.Parameters.arcadia_patch or None
            if patch:
                patch_path = sdk2.svn.Arcadia.apply_patch(
                    arc_path, common_encoding.force_unicode_safe(self.Parameters.arcadia_patch), self.path()
                )
                if patch.startswith("arc:"):
                    patch = patch_path
            self._build_component_binaries(arc_path, patch)
            if not self.Parameters.skip_tests:
                self._run_sandbox_tests(arc_path, arc_url)
            if not self.Parameters.precompile:
                return
            self._prepare_skyboned(arc_path)
            self._precompile_sandbox_sources(arc_path, revision)
            code_info = build_ui.CodeInfoSvn(
                path=os.path.join(arc_path, "sandbox"), branch=branch, rev=revision,
                url=sdk2.svn.Arcadia.trunk_url(path="sandbox", revision=revision)
            )  # use svn url for changelog parsing
            self.Context.sandbox_version = code_info.version
            self.Context.sandbox_revision = code_info.rev
            self.Context.sandbox_changelogs = {}
            components["deploy"] = None
            arcanum_token = sdk2.Vault.data(*Vaults.ARCANUM)
            issue_fetcher = IssueFetcher(arcanum_token)
            revisions = {}
            for component in components:
                components[component] = self.__create_sandbox_resource(code_info, component, components)
                # Strip platform from component (e.g. fileserver-darwin -> fileserver)
                comp_name = component.split("-")[0]
                # Create one changelog per component
                if comp_name in self.Context.sandbox_changelogs:
                    continue
                try:
                    component_changelog = self.create_changelog(
                        code_info,
                        comp_name,
                        archive_type=component,
                        release_tag=self.Parameters.build_type,
                        svn_path_prefix=sdk2.svn.Arcadia.parse_url(code_info.url).path.lstrip("arc"),
                        issue_fetcher=issue_fetcher,
                    )
                    revisions.update(component_changelog["revisions"])
                    self.Context.sandbox_changelogs[comp_name] = component_changelog
                except Exception:
                    logging.exception("Failed to build changelog. %s, archive_type=%s", code_info, comp_name)
            self.Context.revisions = sort_revisions(revisions)
            self.Context.save()

        if self.Parameters.build_type == ctt.ReleaseStatus.STABLE:
            self.Context.start_dc_drills_time, self.Context.end_dc_drills_time = self.get_nearest_dc_drills_event()
            self.Context.save()

            if self.Context.start_dc_drills_time:
                check_dc = self.check_for_dc_drills(self.Context.start_dc_drills_time, self.Context.end_dc_drills_time)

                start_dc_drills_datetime = datetime.datetime.fromtimestamp(self.Context.start_dc_drills_time)
                end_dc_drills_datetime = datetime.datetime.fromtimestamp(self.Context.end_dc_drills_time)

                text = (
                    "<strong>{0}</strong> (<a href=\"{4}\">#{3}</a>)\n"
                    "DC drills are planned from {1} to {2}.\n"
                    "#sandbox_release"
                ).format(
                    check_dc.description,
                    start_dc_drills_datetime,
                    end_dc_drills_datetime,
                    self.id,
                    common_urls.get_task_link(self.id),
                )

                self.set_info("There are DC drills from {} to {}.".format(start_dc_drills_datetime, end_dc_drills_datetime))

                try:
                    self.telegram_bot.send_message(Chats.DEV, text, common_telegram.ParseMode.HTML)
                except Exception:
                    logging.exception("Failed to send notification about DC drills to Telegram")

    def _prepare_skyboned(self, arc_path):
        skyboned_api_dir = self.path("infra/skyboned/api")
        skyboned_api_dir.mkdir(parents=True, exist_ok=True)
        skyboned_api_dir.parent.joinpath("__init__.py").touch()
        skyboned_api_dir.parent.parent.joinpath("__init__.py").touch()
        for f in ("__init__.py", "bencode.py"):
            shutil.copyfile(os.path.join(arc_path, "infra/skyboned/api", f), str(skyboned_api_dir / f))

    def most_recent_check(self):
        """
        Return the most recent CI check
        :rtype: `ci_checks.Check` or None
        """
        # Look for the check with revision less of equal to the one checked out from Arcadia.
        sandbox_revision = int(self.Context.sandbox_revision)
        checks = sorted(ci_checks.recent_checks().values(), key=lambda x: x.revision, reverse=True)
        for check in checks:
            if check.revision <= sandbox_revision:
                return check
        # The 'true' most recent check is not found.
        # Fallback to the most recent revision from the sources included in the release.
        revisions = self.get_revisions()
        if not revisions:
            return
        latest_rev = revisions[0]["revision"]
        return ci_checks.recent_checks([latest_rev]).pop(latest_rev, None)

    @common_patterns.singleton_property
    def infra_client(self):
        infra_token = sdk2.Vault.data(*Vaults.INFRA)
        infra_api = common_rest.Client(InfraConsts.BASE_URL, infra_token)
        return infra_api

    def get_nearest_dc_drills_event(self):
        """
        Returns start and end time of the nearest DC drills event in the next 24 hours from now.
        :rtype: int, int or None, None
        """
        start_dc_drills_time, end_dc_drills_time = None, None

        try:
            dc_drills_from = int(time.time())
            dc_drills_to = dc_drills_from + 24 * 60 * 60

            infra_response = self.infra_client.events.read({"from": dc_drills_from, "to": dc_drills_to, "serviceId": InfraConsts.SERVICE_DC_OUTAGE_ID})

            if infra_response:
                first_event = infra_response[0]
                start_dc_drills_time = first_event.get("start_time")
                end_dc_drills_time = first_event.get("finish_time")

        except common_rest.Client.HTTPError:
            logging.exception("Failed to get Infra event", exc_info=True)
        except common_errors.VaultError:
            logging.exception("Failed to obtain Infra vault token")

        return start_dc_drills_time, end_dc_drills_time

    def validate_release(self):
        if self.Parameters.build_type == ctt.ReleaseStatus.STABLE:
            check = self.most_recent_check()
            if not check:
                raise common_errors.ReleaseError("Could not fetch the most recent CI checks")
            if check.status != ctt.Status.SUCCESS:
                raise common_errors.ReleaseError("Most recent check: r{}:{}".format(check.revision, check.status))
            return check

    @staticmethod
    def check_for_dc_drills(start_event, end_event):
        """
        Checking that release is safe and there are no DC drills in current interval.
        description - More detailed message
        can_release - True/False
        """
        check_dc = collections.namedtuple("event", "description can_release")

        now = int(time.time())
        if not start_event or now > end_event:
            return check_dc("There are no DC drills.", True)

        time_to_event = start_event - now
        if time_to_event <= 0:
            return check_dc("There are DC drills right now!", False)
        elif time_to_event <= 30 * 60:
            return check_dc("There are DC drills less than 30 minutes!", False)
        elif time_to_event <= 6 * 60 * 60:
            return check_dc("There are less than 6 hours left before DC drills. Hurry up for release!", True)
        else:
            return check_dc("There are DC drills within 24 hours, don't forget about the release!", True)

    def on_release(self, additional_parameters):
        build_type = self.Parameters.build_type
        comments, status, subject, releaser = (
            additional_parameters[_].encode("utf-8")
            for _ in ("release_comments", "release_status", "release_subject", "releaser")
        )

        if releaser not in SUPER_RELEASERS:
            self.validate_release()

            self.Context.start_dc_drills_time, self.Context.end_dc_drills_time = self.get_nearest_dc_drills_event()
            self.Context.save()

            check_dc = self.check_for_dc_drills(self.Context.start_dc_drills_time, self.Context.end_dc_drills_time)
            if not (self.Parameters.force_release or check_dc.can_release):
                text = (
                    "<strong>Release cancelled. {0}</strong> (<a href=\"{2}\">#{1}</a>)\n"
                    "#sandbox_release"
                ).format(check_dc.description, self.id, common_urls.get_task_link(self.id))

                try:
                    self.telegram_bot.send_message(Chats.DEV, text, common_telegram.ParseMode.HTML)
                except Exception:
                    logging.exception("Failed to send notification about cancelled release to Telegram")

                raise common_errors.ReleaseError(check_dc.description)

        super(BuildSandbox, self).on_release(additional_parameters)

        # Cache check statuses
        try:
            self.Context.revisions = self.get_revisions(fetch_checks=True)
            self.Context.checks_cached = True
            check = self.most_recent_check()
            if check:
                self.Context.most_recent_check = check._asdict()
        except Exception:
            logging.warning("Could not fetch check statuses", exc_info=True)

        if build_type == ctt.ReleaseStatus.STABLE and self.Context.sandbox_common_subtask:
            self._release_common_package_build()

        if build_type in (ctt.ReleaseStatus.STABLE, ctt.ReleaseStatus.PRESTABLE):
            try:
                event_start = int(time.time())
                event_end = event_start + InfraConsts.EVENT_DURATION

                self.infra_client.events.create(
                    title=subject,
                    description=comments,
                    environmentId=InfraConsts.ENVIRONMENTS[build_type],
                    serviceId=InfraConsts.SERVICE,
                    startTime=event_start,
                    finishTime=event_end,
                    type=InfraConsts.TYPE,
                    severity=InfraConsts.SEVERITY,
                    sendEmailNotifications=False,
                )
            except common_rest.Client.HTTPError:
                logging.exception("Failed to create Infra event", exc_info=True)

            except common_errors.VaultError:
                logging.exception("Failed to obtain Infra vault token")

        # this also skips prestable releases that are getting cancelled
        if build_type != ctt.ReleaseStatus.STABLE:
            return

        # only consider tickets left in formatted changelog
        tickets = set()
        for line in comments.split("\n"):
            match = TICKETS_RE.search(line)
            if match:
                tickets.add(match.group())

        try:
            st_token = sdk2.Vault.data(*Vaults.TRACKER)
        except common_errors.VaultError:
            logging.exception("Failed to obtain Startrek vault token")
            return

        api = common_rest.Client(STARTREK_BASE_URL, st_token)
        for ticket in tickets:
            try:
                api.issues[ticket].comments(
                    text="\n".join((
                        "Shipped with **{rev}**, release task identifier: **(({reference} {id}))**".format(
                            rev=self.Context.sandbox_version,
                            reference=common_urls.get_task_link(self.id),
                            id=self.id
                        ),
                        "++!!(grey)(it takes about half an hour for a release to fully deploy)!!++"
                    )),
                )
                api.issues[ticket]["remotelinks?notifyAuthor=false&backlink=false"](
                    relationship="relates",
                    key=self.id,
                    origin="ru.yandex.sandbox",
                )
            except common_rest.Client.HTTPError:
                logging.exception("Failed to post release notification to %s", ticket)

        text = (
            "<strong>[{1}] {2} by {3}</strong> (<a href=\"{5}\">#{4}</a>)\n"
            "<pre>{0}</pre>\n"
            "#sandbox_release"
        ).format(cgi.escape(comments), status, subject, releaser, self.id, common_urls.get_task_link(self.id))

        for chat_id, send_sticker in ((Chats.DEV, True), (Chats.RELEASES, False)):
            try:
                if send_sticker:
                    stickers = {
                        "aripinen": ["CAACAgIAAxkBAAECDhRidRTC-mUlZ7fsaMcyaoucWmV2CwAC7QEAAmBkLgABAvfOrNmzwiQkBA"],
                        "artanis": ["CAACAgIAAxkBAAECDipileGKc53S7JuV_isp_3uF1EYGuQACpgUAAvBLiwM8n55bwBxfwSQE"],
                        "yetty": [
                            "CAACAgIAAxkBAAECDjFiljZflBHnf0wY0ZAAART2OUx1rRQAAuICAAKZL0sKp5rBxHZLaPQkBA",
                            "CAACAgIAAxkBAAECDjRiljarBN9hsYEPoLCNSUX7LCyLiAAC7AIAApkvSwpGiKtvF89tySQE",
                            "CAACAgIAAxkBAAECDjdiljbFvuQM1yJPpTXBJIrM-yY7bwACewIAApkvSwoYWjRcemSWSSQE",
                            "CAACAgIAAxkBAAECDjpiljbfsoRtPup0PRldiJe9WwRjdAAClAMAApkvSwrjnLbgovvTnSQE",
                            "CAACAgIAAxkBAAECDj1iljcG3AXS9R3XxRgUBdjiUpcHiQACrgMAApkvSwo4vN_RM2A4RCQE",
                        ]
                    }.get(
                        self._releaser(),
                        [common_telegram.Sticker.PIKACHU]
                    )
                    random.seed(time.time())
                    sticker = stickers[random.randint(0, len(stickers) - 1)]
                    self.telegram_bot.send_sticker(chat_id, sticker)
                self.telegram_bot.send_message(chat_id, text, common_telegram.ParseMode.HTML)
            except Exception:
                logging.exception("Failed to send release notes to Telegram")

        logging.debug("Release's additional params: %s", additional_parameters)

    def _releaser(self):
        ctx = self.Context.__CI_CONTEXT or {}
        return ctx.get("flow_triggered_by") or self.author

    def _run_common_package_build(self):
        # Subtask to build sandbox/common/rest package for PyPI
        subtask = self.server.task(
            type="BUILD_SANDBOX_COMMON_PACKAGE",
            children=True,
            owner=self.owner,
            notifications=[{
                'transport': ctn.Transport.TELEGRAM,
                'statuses': ["FAILURE", "BREAK"],
                'recipients': [self.author]
            }],
        )
        self.server.batch.tasks.start.update(subtask["id"])
        self.Context.sandbox_common_subtask = subtask["id"]

    def _release_common_package_build(self):
        try:
            if sdk2.Task[self.Context.sandbox_common_subtask].status == ctt.Status.SUCCESS:
                self.server.release.create({
                    "task_id": self.Context.sandbox_common_subtask,
                    "type": ctt.ReleaseStatus.STABLE,
                    "message": "",
                    "cc": [],
                    "subject": ""
                })
        except Exception:
            logging.warning("Could not release sandbox/common/rest PyPI package", exc_info=True)

    @staticmethod
    def get_svn_revision(arc_path):  # type: (str) -> (str, bool)
        last_checked_commit = None
        counter = 1
        while True:
            logs = arc_module.Arc().log(arc_path, max_count=counter, end_commit=last_checked_commit, as_dict=True)
            for commit in logs:
                if "revision" in commit:
                    return commit["revision"], counter == 1
            counter += 1
            last_checked_commit = commit["commit"]
