import collections

from sandbox.common import rest

Check = collections.namedtuple("Check", ("revision", "task", "status"))


def recent_checks(revisions=None, limit=100):  # type: (list[int], int) -> dict[int, Check]
    """
    Return revision checks from TEST_SANDBOX datebase in TestEnv.
    If `revisions` is None, return `limit` most recent checks.

    >>> recent_checks([5826364])
    {5826364: Check(revision=5826364, task=528437674, status=u'SUCCESS')}

    >>> recent_checks(limit=5).keys()
    [5825811, 5825793, 5826291, 5826364, 5826166]

    :param revisions: return checks for the particular revisions; unknown revisions are ignored
    :type revisions: List[int]
    :param limit: maximum number of returned checks
    :type limit: int
    :rtype: Dict[int, Check]
    """
    kwargs = {}
    if revisions:
        kwargs["hints"] = ["REVISION:{}".format(revision) for revision in revisions]
    try:
        resp = rest.Client().task.read(
            tags=["CI:COMMIT", "SANDBOX::TEST-BUILD_SANDBOX", "JOB-ID:TEST-SANDBOX"], all_tags=1,
            fields="id,status,hints", limit=limit, order="-id",
            **kwargs
        )
    except rest.Client.HTTPError:
        return {}
    checks = {}
    for task in resp.get("items", []):
        hints = task.get("hints", [])
        for hint in hints:
            if hint.startswith("REVISION:"):
                revision = int(hint.replace("REVISION:", ""))
                if revision not in checks:
                    checks[revision] = Check(revision, task["id"], task["status"])
                break
    return checks
