# coding: utf-8
from __future__ import unicode_literals

import os
import json
import shutil
import logging
import collections
import six
if six.PY2:
    import subprocess32 as sp
else:
    import subprocess as sp


from sandbox import sdk2
from sandbox.sdk2 import service_resources

from sandbox.common import fs
from sandbox.common import errors
from sandbox.common import encoding
from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr

from sandbox.projects.common.arcadia import sdk as arcadia_sdk

# Yav secret `BUILD_SANDBOX_COMMON_PACKAGE`
SECRET = (
    "sec-01dp9t0kv2cj0k8gty8veryg1y",
    "ver-01dp9t0kv94fa2grzersw27bjy"
)


class BuildSandboxCommonPackage(sdk2.Task):

    class Requirements(sdk2.Requirements):
        disk_space = 3 * 1024

    class Parameters(sdk2.Parameters):
        description = "Build python package for sandbox/common/rest"
        kill_timeout = 600
        notifications = []

        url = sdk2.parameters.String("Arcadia URL", default="arcadia-arc:/#trunk")

    def on_enqueue(self):
        if not self.Requirements.tasks_resource:
            self.Requirements.tasks_resource = service_resources.SandboxTasksBinary.find(
                attrs={"released": ctt.ReleaseStatus.STABLE, "task_type": self.type.name}
            ).first()

        if not self.Requirements.tasks_resource:
            raise errors.TaskError("Binary task resource is not found")

        self.Requirements.container_resource = BuildSandboxCommonPackageLXC.find(state=ctr.State.READY).first()
        if not self.Requirements.container_resource:
            raise errors.TaskError("LXC container is not found")

    def on_execute(self):

        last_source_hash = None
        last_build = (
            SandboxCommonPackage.find(
                attrs={"released": ctt.ReleaseStatus.STABLE},
                owner=self.owner,
            )
            .order(-sdk2.Resource.id).first()
        )
        if last_build:
            last_source_hash = last_build.source_hash
            logging.info("Latest released build: #%s with hash %s", last_build.id, last_source_hash)

        with arcadia_sdk.mount_arc_path(self.Parameters.url, fallback=True) as path:

            # Build package utility

            ya_path = os.path.join(path, "ya")
            target = os.path.join(path, "sandbox/scripts/build_common_pip")

            with sdk2.helpers.ProcessLog(self, logger="ya_make") as pl:
                sp.check_call([ya_path, "make", target], stdout=pl.stdout, stderr=pl.stderr)

            # Build package

            build_common_pip = os.path.join(target, "build-common-pip")
            with sdk2.helpers.ProcessLog(self, logger="build_common_pip") as pl:
                args = [build_common_pip]
                if last_source_hash:
                    args += ["--last-hash", last_source_hash]
                sp.call(args, stdout=pl.stdout, stderr=pl.stderr)

            # Check result

            try:
                with open("result.json", "r") as f:
                    result = json.load(f)
            except IOError:
                raise errors.TaskFailure("Package build is unsuccesfull: no result found")

            if result["status"] == "unchanged":
                self.set_info("Sources have not been changed since #{}".format(last_build.id))
                return

            if result["status"] != "ok":
                raise errors.TaskFailure("Package build is unsuccesfull: {}".format(result))

            # Create package resource

            package = SandboxCommonPackage(
                task=self,
                description="sandbox/common/rest package",
                source_hash=result["hash"],
                version=result["version"],
                path="./dist"
            )
            resource_data = sdk2.ResourceData(package)
            resource_data.path.mkdir()

            for filename in result["files"]:
                shutil.copy2(str(self.path(filename)), str(resource_data.path))
            resource_data.ready()

            # Test package

            wheel_file = next(self.path().glob('*.whl'), None)
            assert wheel_file is not None

            tox_ini = fs.read_file("sandbox/projects/sandbox/build_sandbox_common_package/tox.ini")
            tox_ini = encoding.force_unicode_safe(tox_ini)
            tox_ini = tox_ini.replace("{wheel}", wheel_file.name)
            (self.path("tox.ini")).write_text(tox_ini)

            (self.path("tests")).mkdir()
            (self.path("tests/__init__.py")).touch(exist_ok=True)
            shutil.copy2(
                os.path.join(path, "sandbox/common/rest/tests/__init__.py"),
                str(self.path("tests/test_rest.py"))
            )

            test_results = self.log_path("test_result.json")

            with sdk2.helpers.ProcessLog(self, logger="tox") as pl:
                test_returncode = sp.call(
                    ["/opt/py3/bin/tox", "-p", "auto", "--result-json", str(test_results)],
                    stdout=pl.stdout, stderr=pl.stderr
                )

            if (self.path("reports")).exists():
                shutil.copytree(str(self.path("reports")), str(self.log_path("reports")))

            if test_results.exists():
                with test_results.open("r") as f:
                    data = json.load(f)

                rows = []

                for key, te in data['testenvs'].items():
                    report_url = None
                    report_path = self.log_path("reports/report-{}.html".format(key))
                    if report_path.exists():
                        report_url = "https://proxy.sandbox.yandex-team.ru/{}/reports/{}".format(
                            self.log_resource.id,
                            report_path.name
                        )

                    rows.append({
                        "name": key,
                        "status": all(cmd['retcode'] == 0 for cmd in te['test']) if te.get('test') else False,
                        "report": report_url,
                    })
                self.Context.test_results = rows

            if test_returncode:
                raise errors.TaskFailure("Some tests have failed, see reports for details")

    @sdk2.footer()
    def footer(self):
        if not self.Context.test_results:
            return
        rows = [
            collections.OrderedDict((
                ("Test", '<a href="{report}">{name}</a>'.format(**test) if test["report"] else test["name"]),
                ("Status", "\u2705" if test["status"] else "\u274c"),
            ))

            for test in self.Context.test_results
        ]
        return {"<h3>Test runs</h3>": rows}

    def on_release(self, params):

        with self.memoize_stage.publish(commit_on_entrance=False):
            package_resource = SandboxCommonPackage.find(task_id=self.id, state=ctr.State.READY).first()
            if not package_resource:
                raise errors.ReleaseError("Could not find the package")
            package = sdk2.ResourceData(package_resource)

            twine_resource = TwineBinary.find(
                state=ctr.State.READY, attrs={"released": ctt.ReleaseStatus.STABLE}
            ).first()
            if not twine_resource:
                raise errors.ReleaseError("Could not find released {} resource".format(TwineBinary.name))
            twine = sdk2.ResourceData(twine_resource)

            files = [str(p) for p in package.path.glob("*")]
            if len(files) != 2:
                raise errors.ReleaseError("Expected two package files, found {}".format(len(files)))

            secrets = sdk2.yav.Secret(*SECRET).data()
            env = os.environ.copy()
            env.update({
                "TWINE_REPOSITORY_URL": "https://pypi.yandex-team.ru/repo/default",
                "TWINE_USERNAME": secrets["pypi_access_key"],
                "TWINE_PASSWORD": secrets["pypi_secret_key"],
            })

            with sdk2.helpers.ProcessLog(self, logger="twine") as pl:
                sp.check_call(
                    [str(twine.path), "upload"] + files,
                    stdout=pl.stdout, stderr=pl.stderr, env=env
                )

            self.set_info("{} has been successfully published to PyPI".format(package_resource.version))

        super(BuildSandboxCommonPackage, self).on_release(params)


class SandboxCommonPackage(sdk2.Resource):
    releasable = True
    auto_backup = True
    restart_policy = ctr.RestartPolicy.DELETE

    source_hash = sdk2.Attributes.String("Source hash", required=True)
    version = sdk2.Attributes.String("Package version", required=True)


class TwineBinary(sdk2.Resource):
    executable = True
    releasable = True
    arcadia_build_path = "junk/mkznts/contrib/twine/twine-bin"


class BuildSandboxCommonPackageLXC(service_resources.LxcContainer):
    pass
