import os
import logging
import tarfile
import datetime as dt

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr

import sandbox.projects.sandbox.resources as sb_resources
from . import builder


class BuildSandboxDocs(sdk2.ServiceTask):
    """
    Build Sandbox documentation for client, server, library, sdk and all tasks' code
    """

    class Parameters(sdk2.Parameters):
        kill_timeout = 6 * 3600
        build_for_tasks = sdk2.parameters.Bool(
            "Build documentation for tasks", description="Adds about 3 hours to build time if run in a single process"
        )

    def _collect_resources(self, dst_dir):
        resources = []
        for resource_type, attrs in common.utils.chain(
            [
                (sb_resources.SandboxArchive, {"type": archive_type, "released": ctt.ReleaseStatus.STABLE})
                for archive_type in ("client", "library", "server")
            ],
            [(sb_resources.SandboxArchive, None)] if self.Parameters.build_for_tasks else []
        ):
            owner = self.owner
            if resource_type == sb_resources.SandboxArchive:
                owner = common.config.Registry().common.service_group
            resource = sdk2.Resource.find(
                type=resource_type,
                owner=owner,
                state=ctr.State.READY,
                attrs=attrs,
            ).order(-sdk2.Resource.id).first()
            resources.append(resource)
            logging.info("Resource of type '%s' with attributes '%s' found: %s", resource_type, attrs, resource)

        revision = 0
        ignored_dirs = []
        for resource in sorted(resources, key=lambda r: r.id):
            archive_path = str(sdk2.ResourceData(resource).path)
            common.fs.untar_archive(archive_path, dst_dir)
            revision = max(
                revision,
                int(getattr(
                    resource,
                    "commit_revision"
                    if isinstance(resource, sb_resources.SandboxTasksArchive)
                    else "version"
                ))
            )

        for root, dirs, files in os.walk(dst_dir):
            for fname in files:
                if fname in ("conftest.py", "conftest.pyc"):
                    path = os.path.join(root, fname)
                    logging.info("Remove file: {}".format(os.path.relpath(path, dst_dir)))
                    os.unlink(path)
            for directory in dirs:
                if directory in ("test", "tests"):
                    path = os.path.join(root, directory)
                    logging.info("Ignore directory: %s", path)
                    ignored_dirs.append(path)
                    continue

        return revision, ignored_dirs

    @staticmethod
    def _gzip_directory(source, destination):
        with tarfile.open(destination, "w:gz") as fh:
            for item in os.listdir(source):
                fh.add(os.path.join(source, item), arcname=item)

    def on_execute(self):
        sphinx_dir = str(self.path("sphinx"))
        source_dir = str(self.path("source"))
        output_dir = str(self.path("output"))

        sandbox_dir = os.path.join(source_dir, "sandbox")
        tasks_dir = os.path.join(source_dir, "projects")

        for d in (sphinx_dir, source_dir, sandbox_dir, tasks_dir):
            os.makedirs(str(d))

        revision, ignored_dirs = self._collect_resources(source_dir)
        release = "r{}".format(revision)

        b = builder.DocsBuilder(
            project_name="Sandbox",
            copyright="{} LLC Yandex".format(dt.date.today().year),
            author="sandbox-dev@",
            version=revision,
            release=release,
            working_dir=sphinx_dir,
            output_dir=output_dir,
            sandbox_dir=sandbox_dir,
            tasks_dir=tasks_dir,
        )

        def make_logger(filename):
            logger = logging.getLogger(filename)
            map(logger.removeHandler, logger.handlers[:])
            handler = logging.FileHandler(str(self.log_path(filename)))
            handler.setFormatter(logging.Formatter("%(asctime)s\t%(message)s"))
            logger.addHandler(handler)
            logger.propagate = False
            return logger

        with sdk2.helpers.ProcessLog(self, logger=make_logger("sphinx-apidoc")) as apidoc_pl:
            with sdk2.helpers.ProcessLog(self, logger=make_logger("sphinx-build")) as build_pl:
                b.build(apidoc_pl.stdout, build_pl.stdout)

        with open(os.path.join(output_dir, ".revision"), "w") as revision_file:
            revision_file.write(str(revision))

        docs_path = self.path("sandbox_docs.tgz")
        self._gzip_directory(output_dir, str(docs_path))

        sdk2.ResourceData(
            sb_resources.SandboxDocs(
                self,
                "Sandbox documentation {}".format(release),
                str(docs_path),
                version=revision,
                ttl="inf",
            )
        )
