from __future__ import absolute_import

import os
import sys
import shutil
import logging
import textwrap
import subprocess as sp


class DocsBuilder(object):
    """
    A piece of code to build Sandbox documentation with sphinx and sphinx.ext.autodoc, which involves
    playing around with sys.path and source directory layout (symbolic links or, altetnatively, copying).
    """

    # see http://www.sphinx-doc.org/en/stable/config.html for details
    CONFIG_TEMPLATE = textwrap.dedent("""
        import os
        import sys
        import inspect


        for path in {paths}:
            sys.path.append(path)

        autodoc_mock_imports = []
        extensions = [
            "sphinx.ext.autodoc",
            "sphinx.ext.todo",
            "sphinx.ext.viewcode"
        ]
        exclude_patterns = [
            "_build", "Thumbs.db", ".DS_Store", "**/conftest.py", "**/resource_types.py"
        ]
        templates_path = ["_templates"]
        source_suffix = ".rst"
        master_doc = "index"

        project = u"{project}"
        copyright = u"{copyright}"
        author = u"{author}"
        version = u"{version}"
        release = u"{release}"
        language = "en"
        todo_include_todos = True

        pygments_style = "sphinx"
        html_theme = "alabaster"
        html_static_path = ["_static"]
        html_sidebars = {{
            "**": [
                "about.html",
                "navigation.html",
                "relations.html",  # needs "show_related": True theme option to display
                "searchbox.html",
            ]
        }}
        htmlhelp_basename = "{project}"


        # SANDBOX-4790
        imported_twice = [
            "sdk2.task",
            "sdk2.resource",
            "sdk2.path",
        ]

        def skip_member(app, what, name, obj, skip, options, already_met=set()):
            if name == "__init__":
                return False
            module = inspect.getmodule(obj)
            if module:
                fqn = ".".join((module.__name__, name))
                if any(fqn.startswith(module_) for module_ in imported_twice):
                    if fqn not in already_met:
                        already_met.add(fqn)
                        return True

            return skip


        def setup(app):
            app.connect("autodoc-skip-member", skip_member)

    """)

    def __init__(
        self, project_name, copyright, author, version, release,
        working_dir, output_dir, sandbox_dir, tasks_dir=None
    ):
        """
        :param project_name: project name to be put as a heading
        :param copyright: copyright string in the footer
        :param author: documentation author
        :param version: project version (commit revision, in this case)
        :param release: release name to be put into <title>
        :param working_dir: Sphinx working directory
        :param output_dir: path to dump doctrees and HTML pages into
        :param sandbox_dir: path to Sandbox code directory
        :param tasks_dir: path to sandbox-tasks directory
        """

        self.sandbox_dir = sandbox_dir
        self.tasks_dir = tasks_dir
        self.working_dir = working_dir
        self.output_dir = output_dir
        self.project_name = project_name
        self.copyright = copyright
        self.author = author
        self.version = version
        self.release = release

    def build(self, apidoc_stdout=None, build_stdout=None):
        source_dir = os.path.join(self.working_dir, "source")
        templates_dir = os.path.join(self.working_dir, "templates")
        html_dir = os.path.join(self.output_dir, "html")
        doctrees_dir = os.path.join(self.output_dir, "doctrees")

        for path in (source_dir, templates_dir, html_dir, doctrees_dir):
            if os.path.exists(path):
                logging.info("Remove %s", path)
                shutil.rmtree(path)
            os.makedirs(path)

        os.symlink(self.sandbox_dir, os.path.join(source_dir, "sandbox"))
        for directory in next(os.walk(self.sandbox_dir))[1]:
            os.symlink(
                os.path.join(self.sandbox_dir, directory),
                os.path.join(source_dir, directory)
            )
        if self.tasks_dir:
            os.symlink(
                self.tasks_dir,
                os.path.join(source_dir, "projects")
            )

        env = os.environ.copy()
        env["PYTHONPATH"] = ":".join([
            "/skynet",
            str(source_dir),
            str(os.path.join(source_dir, "sandbox"))
        ])
        env["PATH"] = ":".join([
            os.path.dirname(sys.executable),
            env.get("PATH", ""),
        ])

        # flags for :automodule: directive
        # see: http://www.sphinx-doc.org/en/stable/ext/autodoc.html#directive-autoattribute
        env["SPHINX_APIDOC_OPTIONS"] = ",".join((
            "imported-members",
            "members",
            "inherited-members",
            "show-inheritance",
            "undoc-members",
        ))

        apidoc_args = [
            sys.executable, "-m", "sphinx.apidoc",  # prefer sphinx-apidoc from our virtual environment
            "-M",  # put module documentation in front of that of submodules
            "-l",  # follow symbolic links
            "-F",  # do a full build, as we still need a master file, index.rst
            "-d", str(1),  # depth of table of contents. more means uglier, especially when tasks' code is included
            "-H", self.project_name,  # project name
            "-A", self.author,  # project author
            "-V", str(self.version),  # project version
            "-R", self.release,  # project release
            "-o", templates_dir,  # directory to write .rst to
            source_dir,  # module/project directory to read sources from
        ]

        build_args = [
            sys.executable, "-m", "sphinx",
            "-b", "html",
            "-d", doctrees_dir,
            "-v",
            templates_dir,
            html_dir,
        ]

        with open(os.path.join(templates_dir, "conf.py"), "w") as conf:
            conf.write(self.CONFIG_TEMPLATE.format(
                paths=["/skynet", source_dir],
                project=self.project_name,
                copyright=self.copyright,
                author=self.author,
                version=self.version,
                release=self.release,
            ))

        sp.Popen(apidoc_args, stdout=apidoc_stdout, stderr=sp.STDOUT, env=env).wait()

        for name in os.listdir(templates_dir):
            if "tests" in name or (name.endswith(".rst") and name.startswith("sandbox.")):
                os.remove(os.path.join(templates_dir, name))

        sp.Popen(build_args, stdout=build_stdout, stderr=sp.STDOUT, env=env).wait()
