import os
import sys
import json
import math
import shutil
import logging
import functools
import compileall
import collections
import threading as th
import multiprocessing as mp

import concurrent.futures

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

from sandbox import sdk2
from sandbox import sandboxsdk
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.common.vcs import arc
from sandbox.projects.common.constants import constants as arcadiasdk_constants
import sandbox.projects.common.build.parameters as build_parameters
import sandbox.projects.common.arcadia.sdk as arcadiasdk


from sandbox.projects.sandbox import resources as sb_resources
from sandbox.projects.sandbox import common as sandbox_common
from sandbox.projects.sandbox import remote_copy_resource
from sandbox.projects.sandbox.common import computation_graph


TASKS_BINARY_TARGET = "sandbox/tasks/tests/bin"
SKYNET_PATH = "/skynet"


def measure_time(group, parallel=False):
    """
    Record function's execution duration in the task's context
    :param group: name of the stage this function belongs to
    :param parallel: count stage duration as max(durations) instead of sum()
    """

    def wrapper(f):
        @functools.wraps(f)
        def inner(task, *a, **kw):
            assert task is sdk2.Task.current, "Can only work within current task's context!"
            with common.utils.Timer() as timer:
                result = f(task, *a, **kw)
            with th.RLock():
                group_, duration, parallel_ = task.Context.stages.get(f.__name__, (group, 0, parallel))
                task.Context.stages[f.__name__] = (group_, duration + int(timer.secs), parallel_)

            return result

        return inner

    return wrapper


class SourcePackaging(common.utils.Enum):
    RAW = None
    BINARY = None


class BuildSandboxTasks(sdk2.ServiceTask):
    """
    Test and build Sandbox tasks bundle.
    Add `.revision` file to each directory in `projects`.
    Embed preprocessed ReST documentation into each task module.
    """

    BLACKLIST_INITS = {
        "sandbox/projects/geobase/common/LAAS-964.hosting+proxy+vpn-fix",
        "sandbox/projects/tank/offlinereport",
        "sandbox/projects/ToolsCaasConfigure/ut",
        "sandbox/projects/UrlsByShowCounters/executable",
        "sandbox/projects/geobase/common/LIB-795",
        "sandbox/projects/logs/scarab/DeployScarabSwiftApi/updateSwiftScript",
    }

    FILE_MOVES = {
        "sandbox/projects/geobase/common/export/export2yt.py": ["sandbox/projects/geobase/common/export2yt.py"],
        "sandbox/projects/adfox/counters_v2/counters_base/counters_base_class.py": [
            "sandbox/projects/adfox/counters_v2/counters_base/counters_base_class.py",
            "sandbox/projects/adfox/counters_v2/tasks/ch_update_daily_counters/counters_base_class.py",
            "sandbox/projects/adfox/counters_v2/tasks/ch_update_hourly_counters/counters_base_class.py",
            "sandbox/projects/adfox/counters_v2/tasks/merge_daily_counters_into_total/counters_base_class.py",
            "sandbox/projects/adfox/counters_v2/tasks/merge_hourly_counters_into_daily/counters_base_class.py"
        ],
        "yql/udfs/yql_deploy_udfs.yaml": ["sandbox/projects/yql/YQLBatchDeployUDF/udfs_config.yaml"]
    }

    FILE_PREF = "$S/"

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.LINUX_FOCAL & (ctc.Tag.LXC | ctc.Tag.PORTOD)
        cores = 16
        ram = 64 * 1024
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 8 * 1024, None)
        dns = ctm.DnsType.DNS64
        disk_space = 8 << 10
        environments = (
            sandboxsdk.environments.PipEnvironment("toml", "0.10.0", use_wheel=True),
        )

        class Caches(sdk2.Requirements.Caches):
            pass  # Do not use any caches to run on multislot

    class Parameters(sdk2.Parameters):
        kill_timeout = 60 * 60  # support worst case scenario (local build on a revision for which there's no cache)

        tasks_svn_url = sdk2.parameters.ArcadiaUrl(
            default_value=sdk2.svn.Arcadia.trunk_url(), required=True
        )
        use_last_binary_tasks_resource = sdk2.parameters.Bool("Use last binary sandbox tasks resource", default=False)
        auto_deploy = sdk2.parameters.Bool("Add auto deploy attribute to resources", default=False, do_not_copy=True)
        auto_deploy_to_preproduction = sdk2.parameters.Bool(
            "Push resources to preproduction", default=False, do_not_copy=True
        )
        arcadia_patch = sdk2.parameters.String(
            build_parameters.ArcadiaPatch.description,
            default=build_parameters.ArcadiaPatch.default,
            multiline=True
        )  # `consts.ARCADIA_PATCH_KEY`

        with sdk2.parameters.RadioGroup("Sources packaging method") as packaging:
            packaging.values[SourcePackaging.RAW] = packaging.Value("Checkout from Arcadia", default=True)
            packaging.values[SourcePackaging.BINARY] = packaging.Value("Extract from binary")

        with packaging.value[SourcePackaging.BINARY]:
            with sdk2.parameters.String(
                "Build mode", description=(
                    "Most of the time you want semi-distbuild mode, as it is the most reliable and fastest. "
                    "Use ya make for testing purposes and distbuild for when you're short of system resources"
                )
            ) as build_mode:
                build_mode.values[arcadiasdk_constants.YMAKE_BUILD_SYSTEM] = build_mode.Value("ya make (local run)")
                build_mode.values[arcadiasdk_constants.SEMI_DISTBUILD_BUILD_SYSTEM] = build_mode.Value(
                    "ya make + distbuild (whichever comes first)", default=True
                )
                build_mode.values[arcadiasdk_constants.DISTBUILD_BUILD_SYSTEM] = build_mode.Value("distbuild only")

            use_yt_cache = sdk2.parameters.Bool("Use YT cache for local build", default=False)
            ramdrive_cache = sdk2.parameters.Bool("Put build cache in TMPFS (RAM drive)", default=True)

        with sdk2.parameters.Output:
            bundle = sdk2.parameters.Resource("Resource with tasks bundle (.tar.gz)", required=True)
            image = sdk2.parameters.Resource("Resource with tasks image (.squashfs)", required=True)
            revision = sdk2.parameters.Integer("Latest commit revision for tasks bundle", required=True)

    class Context(sdk2.Context):
        stages = {}
        __do_not_dump_ramdrive = True

    PREPROD_API_URL = "https://www-sandbox1.n.yandex-team.ru/api/v1.0"
    PREPROD_TIMEOUT = 30
    PREPROD_TOKEN_NAME = "preprod-oauth-token"

    TASK_TESTS_TIMEOUT = 900
    COMMON_TESTS_TIMEOUT = 300

    ARCADIA_GROUPS_PATH = "arcadia:/arc/trunk/arcadia/groups"
    ARCADIA_YASDK_PATH = "arcadia:/arc/trunk/arcadia/devtools/ya"

    def make_file_logger(self, logger_name, level=logging.DEBUG):
        logger = logging.getLogger(logger_name)
        map(logger.removeHandler, logger.handlers[:])

        fp = self.log_path("{}.log".format(logger_name))
        handler = logging.StreamHandler(open(str(fp), "a"))
        handler.vault_filter = common.log.VaultFilter()
        handler.addFilter(common.log.TimeDeltaMeasurer())
        handler.addFilter(handler.vault_filter)
        handler.setFormatter(logging.Formatter(ctt.TASK_LOG_FORMAT))
        handler.setLevel(level)

        logger.addHandler(handler)
        logger.setLevel(level)
        logger.propagate = False
        return logger

    def on_save(self):
        self.Parameters.auto_deploy_to_preproduction = (
            self.Parameters.auto_deploy_to_preproduction or
            (
                self.Parameters.auto_deploy and
                self.owner == common.config.Registry().common.service_group
            )
        )

    def on_enqueue(self):
        if self.Parameters.use_last_binary_tasks_resource:
            self.Requirements.tasks_resource = sdk2.Resource[sdk2.service_resources.SandboxTasksBinary].find(
                state=ctr.State.READY, owner="SANDBOX", attrs={"auto_deploy": True, "task_type": str(self.type)}
            ).first()
        auto_deploy_group = common.config.Registry().common.service_group
        if self.Parameters.auto_deploy and self.owner != auto_deploy_group:
            raise common.errors.TaskFailure(
                "Only users that belong to group {!r} can set 'auto_deploy' in task context".format(
                    auto_deploy_group
                )
            )

        if self.Parameters.ramdrive_cache:
            self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 8 * 1024, None)

    def run_collector_script(self, l_env, description_path, groups_dir):
        logging.info("Preparing descriptions cache")

        yasdk_dir = self.path("yasdk")
        sdk2.svn.Arcadia.export(self.ARCADIA_YASDK_PATH, yasdk_dir, revision=3339318)
        l_env["PYTHONPATH"] = ":".join(
            map(str, (self.stable_arcadia_dir, self.stable_arcadia_dir / "sandbox", yasdk_dir, l_env["PYTHONPATH"]))
        )

        script_path = sdk2.Path(__file__).parent / "tasks_info_collector" / "lib" / "main.py"
        logging.info("Launching tasks info collector: '%s'", script_path)
        with sdk2.helpers.ProcessLog(self, logger="tasks-info-collector") as pl:
            # FIXME: SANDBOX-4331
            commandline = [
                sys.executable, str(script_path),
                "--groups", str(groups_dir),
                "--projects", str(self.projects_dir)
            ]
            stdout = sp.check_output(commandline, env=l_env, stderr=pl.stderr).decode("utf-8")

        logging.info("Parsing output of %s length as JSON", common.utils.size2str(len(stdout)))
        # Check that description json can be loaded
        json.loads(stdout)
        logging.info("Dumping descriptions data to '%s'", description_path)
        description_path.write_text(stdout)

    @measure_time("Testing and processing", parallel=True)
    def description_and_owners_collector(self, l_env, description_path):
        groups_dir = self.path("groups")
        sdk2.svn.Arcadia.export(self.ARCADIA_GROUPS_PATH, groups_dir)

        if common.system.inside_the_binary():
            from sandbox.projects.sandbox.build_sandbox_tasks.tasks_info_collector.lib import collector
            collector.store_owners(str(groups_dir), str(self.projects_dir))
        else:
            self.run_collector_script(l_env, description_path, groups_dir)

        # Extracted sources may be dirty in case existing directory is supplied with --target argument,
        # so we need to clean it up
        if self.Parameters.packaging == SourcePackaging.BINARY:
            sandbox_dir = str(self.sandbox_dir)
            exclusions = {
                ".revision",  # revision of arcadia/sandbox
                "projects",  # tasks code directory
            }
            for p in filter(lambda item: item not in exclusions, os.listdir(sandbox_dir)):
                full_path = os.path.join(sandbox_dir, p)
                remove = shutil.rmtree if os.path.isdir(full_path) else os.unlink
                remove(full_path)

    @staticmethod
    def get_svn_revision(arc_path):  # type: (str) -> (str, bool)
        last_checked_commit = None
        counter = 1
        while True:
            logs = arc.Arc().log(arc_path, max_count=counter, end_commit=last_checked_commit, as_dict=True)
            for commit in logs:
                if "revision" in commit:
                    return commit["revision"], counter == 1
            counter += 1
            last_checked_commit = commit["commit"]

    @measure_time("Testing and processing", parallel=True)
    def tasks_revisions(self, is_patched):
        """Set tasks revision equal to the whole 'projects' revision"""
        logger = self.make_file_logger("tasks-revisions")
        for dir_node in self.projects_dir.glob("**"):
            dir_name = dir_node.relative_to(self.projects_dir)
            if str(dir_name).startswith(".svn"):
                continue
            revision = (u"-" if is_patched else u"") + str(self.Parameters.revision)
            logger.debug("Projects dir '%s' revision %r", dir_name, revision)
            (dir_node / ".revision").write_text(revision)

    @sdk2.header()
    def head(self):
        if self.status in (ctt.Status.FAILURE, ctt.Status.EXCEPTION) and self.Context.html_report_view:
            return '<span style="font-size: 15px">{}</span>'.format(self.Context.html_report_view)

    def append_inits(self, path):
        init_file = False
        py_file = False
        if path[len(str(self.arcadia_dir)):].strip("/") in self.BLACKLIST_INITS:
            return False
        for name in os.listdir(path):
            full_name = os.path.join(path, name)
            if os.path.isdir(full_name):
                py_file = (self.append_inits(full_name) or py_file)
            else:
                if name == "__init__.py":
                    init_file = True
                    py_file = True
                elif name.endswith(".py"):
                    py_file = True

        if py_file and not init_file:
            open(os.path.join(path, "__init__.py"), "a").close()
        return py_file

    def emulate_binary_build(self, arcadia_path):  # type: (str) -> None
        ya_bin = os.path.join(arcadia_path, "ya")
        dump_path = os.path.join(arcadia_path, "sandbox", "tasks", "tests", "bin")
        files_cmd = [self.venv_python, ya_bin, "dump", "files", "--ignore-recurses", dump_path]
        with sdk2.helpers.ProcessLog(self, logger="ya_files") as pl:
            stdout = sp.check_output(files_cmd, stderr=pl.stderr).decode("utf-8")
        lines = [line.strip().split(" ") for line in stdout.split("\n")]
        sandbox_files = []
        for line in lines:
            if len(line) > 2 and line[1] == "File":
                path = line[-1][len(self.FILE_PREF):]
                if path.startswith("sandbox"):
                    sandbox_files.append(path)

        def copy_file(src, dst):  # type: (str, str) -> None
            dist_path = os.path.join(str(self.arcadia_dir), dst)
            if not os.path.exists(os.path.dirname(dist_path)):
                os.makedirs(os.path.dirname(dist_path))
            shutil.copy(os.path.join(arcadia_path, src), dist_path)

        for file in sandbox_files:
            copy_file(file, file)
        for src, dsts in self.FILE_MOVES.items():
            for dst in dsts:
                copy_file(src, dst)

        self.append_inits(str(self.sandbox_dir))

    @measure_time("Fetching code")
    def fetch_source_code(self):  # type: () -> str
        with arcadiasdk.mount_arc_path(self.Parameters.tasks_svn_url, use_arc_instead_of_aapi=True) as arc_path:
            target_revision, is_latest_commit = self.get_svn_revision(arc_path)
            svn_url_info = sdk2.svn.Arcadia.info(
                sdk2.svn.Arcadia.trunk_url("sandbox/projects", revision=target_revision)
            )
            projects_revision = svn_url_info["commit_revision"]
            logging.info(
                "Build sandbox tasks from %s, revision %s%s",
                self.Parameters.tasks_svn_url,
                projects_revision,
                " (with patch)" if self.Parameters.arcadia_patch or not is_latest_commit else ""
            )
            if self.Parameters.arcadia_patch:
                sdk2.svn.Arcadia.apply_patch(arc_path, self.Parameters.arcadia_patch, self.path())
            if self.Parameters.packaging == SourcePackaging.BINARY:
                self.emulate_binary_build(arc_path)
                self.copy_ya_make_files(arc_path)
            else:
                os.makedirs(str(self.sandbox_dir))
                shutil.copytree(os.path.join(arc_path, "sandbox", "projects"), str(self.projects_dir))
                # sandbox.tasks.tests.lib module is required by tests
                shutil.copytree(os.path.join(arc_path, "sandbox", "tasks"), str(self.tasks_dir))
                self.remove_py3_sources()

        # fetch all packages to one folder
        release_type = common.config.Registry().server.services.packages_updater.release_status
        try:
            resources = {}
            for kind in ("server", "server-tests", "client"):
                res = sb_resources.SandboxArchive.find(
                    attrs=dict(released=release_type, type=kind)
                ).order(-sdk2.Task.id).limit(1).first()
                if not res:
                    raise ValueError("No {!r} released resource of {!r} type.".format(release_type, kind))
                resources[sdk2.ResourceData(res).path] = res
            logging.info("Ready Sandbox resources found: %r", resources)
            for archive_path, _ in sorted(resources.items(), key=lambda kv: int(kv[1].version)):
                common.fs.untar_archive(str(archive_path), str(self.stable_arcadia_dir))

        except Exception as ex:
            logging.warning("Cant get resource: %s. Try checkout from SVN", ex)
            shutil.rmtree(str(self.stable_arcadia_dir), ignore_errors=True)
            # if sandbox package not found then get it from subversion
            core_dir = self.stable_arcadia_dir / "sandbox"
            sdk2.svn.Arcadia.checkout(sdk2.svn.Arcadia.trunk_url("sandbox"), core_dir)
            # drop tasks code from sandbox core directory and pytest stuff
            shutil.rmtree(str(core_dir / "projects"), ignore_errors=True)
            map(sdk2.path.Path.unlink, (core_dir / "pytest.ini", core_dir / "conftest.py"))

        if not self.projects_dir.exists():
            raise common.errors.TaskFailure("Dir '{!s}' does not exist.".format(self.projects_dir))

        (self.stable_arcadia_dir / "sandbox" / "projects").symlink_to(self.projects_dir)
        rev_path = self.sandbox_dir / ".revision"
        rev_path.write_text(unicode(projects_revision))
        return projects_revision

    def copy_ya_make_files(self, arc_path):  # type: (str) -> None
        arc_projects_dir = os.path.join(arc_path, "sandbox", "projects")
        for root, _, files in os.walk(arc_projects_dir):
            for filename in files:
                if filename == "ya.make":
                    full_path = os.path.join(root, filename)
                    relative_source = os.path.relpath(full_path, arc_projects_dir)
                    destination = os.path.join(str(self.projects_dir), relative_source)
                    try:
                        shutil.copyfile(full_path, destination)
                    except IOError:  # intermediate directories may be missing
                        pass

    def remove_py3_sources(self):  # type: () -> None
        py3_sources_binary_res = sb_resources.SandboxPy3ModulesParser.find(
            owner="SANDBOX",
            state=ctr.State.READY,
            attrs=dict(released=ctt.ReleaseStatus.STABLE)
        ).order(-sdk2.Task.id).limit(1).first()
        if not py3_sources_binary_res:
            raise ValueError("Resource of type {} not found.".format(str(sb_resources.SandboxPy3ModulesParser)))
        py3_sources_binary = str(sdk2.ResourceData(py3_sources_binary_res).path)
        py3_modules = [
            module.replace(".", os.sep)
            for module in sp.check_output([py3_sources_binary, "-r", str(self.projects_dir)]).strip().split(",")
        ]
        for module in py3_modules:
            try:
                if module:
                    shutil.rmtree(str(self.projects_dir / module))
            except OSError:
                logging.info("Can't remove python3 directory %s", self.sandbox_dir / module)

    @property
    def client_venv(self):
        return os.path.join(common.os.User.service_users.service.home, "venv")

    @property
    def venv_python(self):
        return os.path.join(self.client_venv, "bin", "python")

    @measure_time("Testing and processing", parallel=True)
    def common_tests(self, env):
        logging.info("Test Sandbox tasks.")
        html_report = sdk2.ResourceData(self.log_resource).path / "report.html"
        num_processes = int(math.ceil(mp.cpu_count() / 2.0))

        processes_to_wait = []
        venv_pytest = os.path.join(self.client_venv, "bin", "pytest")
        logging.info("Venv path for tests: %s", venv_pytest)

        if self.Parameters.packaging == SourcePackaging.BINARY:
            env = os.environ.copy()
            env["PYTHONPATH"] = SKYNET_PATH
            logging.debug("Test subprocess environment: %r", env)
            pl = sdk2.helpers.ProcessLog(self, logger="test-tasks-sandbox-by-binary")
            processes_to_wait.append((
                sp.Popen(
                    [
                        self.venv_python,
                        venv_pytest,
                        "-s", "-v", "-n", str(num_processes), "--run-long-tests",
                        "--confcutdir={}".format(self.stable_arcadia_dir / "sandbox" / "projects"),
                        "--html={}".format(html_report), "--self-contained-html",
                        str(self.stable_arcadia_dir / "sandbox" / "projects")
                    ],
                    stdout=pl.stdout, stderr=pl.stderr, env=env
                ),
                pl, self.TASK_TESTS_TIMEOUT
            ))

        else:
            cmd_pattern = (
                "import sys, pytest; "
                "sys.exit(pytest.main(['-s', '-v', '-n', '{}', '--run-long-tests', '--confcutdir={!s}', '{!s}'{}]))"
            )

            tasks_dir = str(self.sandbox_dir)
            for tests_root_dir in (self.projects_dir, self.projects_dir / "tests"):
                pl = sdk2.helpers.ProcessLog(self, logger="test-tasks-sandbox")
                html_report_suffix = ""
                if tests_root_dir == self.projects_dir:
                    html_report_suffix = ", '--html={!s}', '--self-contained-html'".format(html_report)
                processes_to_wait.append((
                    sp.Popen(
                        [self.venv_python, "-c", cmd_pattern.format(
                            num_processes, self.projects_dir, tests_root_dir, html_report_suffix
                        )],
                        env=env, stdout=pl.stdout, stderr=pl.stderr, cwd=tasks_dir
                    ),
                    pl, self.TASK_TESTS_TIMEOUT
                ))

            # run tests from common/tests
            cmd = [
                self.venv_python,
                str(self.projects_dir / "common" / "tests" / "main.py"),
                "--sandbox_dir=" + str(self.stable_arcadia_dir),
                "--tasks_dir=" + str(self.sandbox_dir),
                "--temp_dir=" + str(self.path("common-tests-tmp"))
            ]
            pl = sdk2.helpers.ProcessLog(self, logger="test-tasks-common")
            processes_to_wait.append((sp.Popen(
                cmd, stdout=pl.stdout, stderr=pl.stderr, cwd=str(self.sandbox_dir)
            ), pl, self.COMMON_TESTS_TIMEOUT))

        def _error_checker(p, pl, timeout, errors):
            try:
                p.wait(timeout)
                pl.raise_for_status(p)
            except (sp.CalledProcessError, sp.TimeoutExpired) as ex:
                logging.exception("Error in subprocess %r", p)
                errors.append(ex)
            finally:
                pl.close()

        errors = []
        checkers = [th.Thread(target=_error_checker, args=args + (errors,)) for args in processes_to_wait]
        map(th.Thread.start, checkers)
        logging.info("Waiting for subprocesses")
        map(th.Thread.join, checkers)

        def _info_setter(ex):
            try:
                self.set_info(ex.get_task_info(), do_escape=False)
            except AttributeError:
                pass

        map(_info_setter, errors)
        if errors:
            if html_report.exists():
                html_view = sdk2.helpers.gdb.get_html_view_for_logs_file(
                    "Tests have failed. See the report for details", "report.html", self.log_resource
                )
                self.set_info(html_view, False)
                self.Context.html_report_view = html_view
            raise common.errors.TaskFailure(
                "There are {} error(s) occurred during bundle build (see logs above)".format(len(errors))
            )

    def modify_and_test_code(self, is_patched):  # type: (bool) -> None
        env = dict(os.environ)
        env["PYTHONPATH"] = ":".join((str(self.stable_arcadia_dir), env.get("PYTHONPATH", "")))

        l_env = dict(os.environ)
        l_env.pop("PWD", None)
        l_env.pop("OLDPWD", None)
        l_env.pop("SANDBOX_CONFIG", None)
        l_env["PYTHONPATH"] = ":".join((SKYNET_PATH, env.get("PYTHONPATH", "")))

        comp_graph = computation_graph.SimpleComputationGraph()

        comp_graph.add_job(
            self.tasks_revisions, args=(is_patched,)
        )

        description_path = self.path(".descriptions")
        comp_graph.add_job(
            self.description_and_owners_collector,
            args=(l_env.copy(), description_path)
        )

        comp_graph.add_job(
            self.common_tests,
            args=(env,),
        )

        comp_graph.run()

    @measure_time("Pushing to preproduction")
    def push_to_preprod(self):
        try:
            preprod_token = sdk2.Vault.data(self.owner, self.PREPROD_TOKEN_NAME)
        except common.errors.VaultError:
            logging.error("Failed to get preprod OAuth token", exc_info=True)
            return

        api = common.rest.Client(self.PREPROD_API_URL, preprod_token, total_wait=self.PREPROD_TIMEOUT)
        last_archive = next(iter(api.resource.read(
            type=sb_resources.SandboxTasksArchive.name,
            state=ctr.State.READY,
            owner=self.owner,
            attrs={"auto_deploy": True},
            limit=1
        )["items"]), None)
        if last_archive:
            actual = int(last_archive["attributes"].get("commit_revision", 0))
            if actual >= self.Parameters.revision:
                logging.info(
                    "Tasks archive version on preprod, %d, is newer than the current one, %d, skipping sync",
                    actual, self.Parameters.revision
                )
                return

        attributes = ["commit_revision={}".format(self.Parameters.revision)]
        if self.Parameters.auto_deploy:
            attributes.append("auto_deploy=True")

        tasks = []
        for res, package_type in (
            (self.Parameters.bundle, "Archive with tasks"),
            (self.Parameters.image, "SquashFS image with tasks"),
        ):
            tasks.append(api.task(
                type=remote_copy_resource.RemoteCopyResource.type,
                owner=self.owner,
                description="{} r{} obtained from <b>{}</b> by <a href='{}'>#{}</a>".format(
                    package_type,
                    self.Parameters.revision,
                    "binary" if self.Parameters.packaging == SourcePackaging.BINARY else "VCS",
                    common.utils.get_task_link(self.id),
                    self.id,
                ),
                custom_fields=[
                    {
                        "name": "resource_type",
                        "value": str(res.type),
                    },
                    {
                        "name": "created_resource_name",
                        "value": str(res.path),
                    },
                    {
                        "name": "remote_file_name",
                        "value": res.skynet_id,
                    },
                    {
                        "name": "resource_attrs",
                        "value": ",".join(attributes),
                    },
                ],
                kill_timeout=5 * 60
            )["id"])
        api.batch.tasks.start = {
            "comment": "Actualize tasks code on preprod",
            "id": tasks
        }
        logging.info("Tasks #%r enqueued on preprod", tasks)

    @property
    def tasks_binary(self):
        return str(self.path(TASKS_BINARY_TARGET, "test-tasks"))

    @common.utils.singleton_property
    def arcadia_dir(self):
        """
        Arcadia root on tmpfs. Tasks code, which is stored somewhere inside, is later packed into an archive
        """
        fake_arcadia_root = self.ramdrive.path / "arcadia"
        fake_arcadia_root.mkdir(parents=True, exist_ok=True)
        return fake_arcadia_root

    @common.utils.singleton_property
    def sandbox_dir(self):
        """
        Where "projects" directory is located
        """
        return self.arcadia_dir / "sandbox"

    @common.utils.singleton_property
    def projects_dir(self):
        """
        Tasks code directory itself (aka "projects")
        """
        return self.sandbox_dir / "projects"

    @common.utils.singleton_property
    def tasks_dir(self):
        """
        Tasks tests directory (aka "tasks")
        """
        return self.sandbox_dir / "tasks"

    @common.utils.singleton_property
    def stable_arcadia_dir(self):
        """
        Copy of Arcadia where latest stable Sandbox code is extracted to
        """
        return self.path("stable_arcadia")

    @common.utils.singleton_property
    def build_cache_dir(self):
        if self.Parameters.ramdrive_cache:
            return self.ramdrive.path / "build_cache_dir"
        else:
            return self.path("build_cache_dir")

    def make_archive(self, sandbox_dir, exclude, description):
        projects_arc_path = self.path("sandbox-tasks.r{}.tar.gz".format(self.Parameters.revision))
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("tar.gz")) as pl:
            cmd = ["tar"]
            cmd += list(common.utils.chain(*(("--exclude", _) for _ in exclude)))
            cmd += ["-C", sandbox_dir, "-zcf", str(projects_arc_path), "."]
            sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr)

        self.Parameters.bundle = sb_resources.SandboxTasksArchive(
            self,
            description,
            projects_arc_path,
            commit_revision=self.Parameters.revision
        )
        sdk2.ResourceData(self.Parameters.bundle).ready()

    def make_image(self, sandbox_dir, exclude, description):
        logging.info("Create Sandbox tasks image resource asynchronously")
        projects_img_path = self.path("sandbox-tasks.r{}.squashfs".format(self.Parameters.revision))
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("squashfs")) as pl:
            cmd = [
                "mksquashfs", sandbox_dir, str(projects_img_path),
                "-no-progress", "-comp", "lzo", "-no-recovery", "-wildcards"
            ]
            cmd += list(common.utils.chain(*(("-e", "... " + _) for _ in exclude)))
            sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr)

        self.Parameters.image = sb_resources.SandboxTasksImage(
            self,
            description,
            projects_img_path,
            commit_revision=self.Parameters.revision
        )
        sdk2.ResourceData(self.Parameters.image).ready()

    @measure_time("Packaging source code")
    def pack_tasks_code(self):  # type: () -> None
        logging.info("Create Sandbox tasks archive resource asynchronously")
        description = "Sandbox tasks, rev. %s, from %s" % (self.Parameters.revision, self.Parameters.tasks_svn_url)
        sandbox_common.embed_revision(self.projects_dir / "__init__.py", self.Parameters.revision)

        sandbox_dir = str(self.sandbox_dir)
        exclude = ("__pycache__", ".cache", ".svn")
        compileall.compile_dir(sandbox_dir, force=True)

        with concurrent.futures.ThreadPoolExecutor(max_workers=2) as pool:
            futures = [
                pool.submit(fn, sandbox_dir, exclude, description)
                for fn in (self.make_image, self.make_archive)
            ]
            for future in concurrent.futures.as_completed(futures):
                exc = future.exception()
                if exc is not None:
                    raise exc

        if self.Parameters.auto_deploy:
            self.Parameters.bundle.auto_deploy = self.Parameters.image.auto_deploy = True

    def on_execute(self):
        self.Parameters.revision = self.fetch_source_code()
        self.modify_and_test_code(is_patched=bool(self.Parameters.arcadia_patch))
        self.pack_tasks_code()
        try:
            if self.Parameters.auto_deploy_to_preproduction:
                self.push_to_preprod()
            else:
                logging.info(
                    "Parameter 'auto_deploy_to_preproduction' is False. Resources will not be sent to preproduction."
                )
        except Exception:
            logging.error("Could not update tasks code on preprod", exc_info=True)

    @sdk2.report(title="Stages timing")
    def stages_duration(self):
        if not self.Context.stages:
            return None

        stages = collections.OrderedDict()
        for function_name, (group, duration, parallel) in self.Context.stages.items():
            stages.setdefault(group, []).append((function_name, duration, parallel))

        def details(group_):
            durations_ = stages[group_]
            parallel_ = durations_[0][2]  # if the first call is parallel, consider the rest parallel as well
            aggregator = max if parallel_ else sum
            head = "<span class='status status_success'>{}</span> (parallel: {}): {}s".format(
                group_, parallel_, aggregator(_[1] for _ in durations_)
            )
            breakdown = "".join(
                "<li><span class='status status_executing'>{}</span>: {}s</li>".format(fn, duration_)
                for (fn, duration_, _) in durations_
            )
            return "{}<br><ul>{}</ul>".format(head, breakdown)

        return "<br>".join(details(group_) for group_ in stages)
