# coding: utf-8

from __future__ import print_function

import os
import sys
import logging

from sandbox import common
from sandbox.common.types import misc as ctm
from sandbox.sdk2.helpers import misc as helpers

import toml
import yasdk.sdk


def source_name(fn):
    return fn[:-1] if fn.endswith(".pyc") else fn


def trim_projects_dir(path):
    """trims the prefix before the first 'project' directory in the path"""
    path_items = path.split(os.sep)
    if "projects" not in path_items:
        return path
    return os.sep.join(path_items[path_items.index("projects"):])


def collect_owners(rb_data, projects_path):
    base_task_dir = projects_path
    owners_path = os.path.join(projects_path, ctm.EnvironmentFiles.OWNERS_STORAGE)
    storage = helpers.SingleFileStorage(owners_path)
    for path, _, files in os.walk(base_task_dir):
        if "ya.make" in files:
            trimmed_path = trim_projects_dir(path)
            storage[trimmed_path] = get_class_owners(path, rb_data)
    return storage


def get_class_owners(ya_make_dir, rb_data):
    yamake_file_path = os.path.join(ya_make_dir, "ya.make")
    if os.path.isfile(yamake_file_path):
        logins_and_groups = yasdk.sdk.owners_logins_and_rbgroups(yamake_file_path)
        groups = logins_and_groups["rbgroups"]
        groups_members = []
        for group in groups:
            groups_members.extend(rb_data.get(group, []))
        return sorted(list(set(logins_and_groups["logins"] + groups_members)))


def get_class_info(cls):
    file_path = source_name(sys.modules[cls.__module__].__file__)
    return {
        "class_name": cls.__name__,
        "file_path": file_path,
        "description": cls.description,
    }


def get_task_class_info(task_cls, owners_storage):
    res = get_class_info(task_cls)
    res["type"] = task_cls.type
    # TBD: eventually, owners shouldn't be saved in class info at all
    res["owners"] = owners_storage[trim_projects_dir(res["file_path"])]
    return res


def parse_review_groups(groups_dir):
    groups = {}
    for group in os.listdir(groups_dir):
        # this will hopefully save us from files that are NOT review groups
        group_file_name = os.path.join(groups_dir, group)
        if os.path.islink(group_file_name) or "." in group or not os.path.isfile(group_file_name):
            continue

        with open(group_file_name, "r") as fd:
            try:
                group_data = toml.loads(fd.read())
                groups[group] = group_data.get("members", [])
            except:
                groups[group] = []

    return groups


def store_owners(groups_dir, projects_path, logger=None):
    logger = logger or logging
    rb_data = parse_review_groups(groups_dir)

    owners_storage = collect_owners(rb_data, projects_path)
    try:
        owners_storage.save()
    except IOError as ex:
        logger.error("Can't save owners: %s", ex)
    return owners_storage


def parse_tasks_code(groups_dir, tasks_dir=None, logger=None, projects_path=None):
    from sandbox.projects import resource_types

    logger = logger or logging
    common.config.Registry().server.web.rst2html = True

    kwargs = {}
    if tasks_dir is not None:
        kwargs = dict(tasks_dir=tasks_dir)

    task_types = common.projects_handler.load_project_types(**kwargs)
    tasks = []
    owners_storage = store_owners(groups_dir, projects_path, logger)

    for cls_info in task_types.itervalues():
        try:
            tasks.append(get_task_class_info(cls_info.cls, owners_storage))
        except KeyError as ex:
            logger.error("Can't process task %r metadata: %s", cls_info, ex)

    resources = []
    for res_cls in resource_types.AbstractResource:
        try:
            resources.append(get_class_info(res_cls))
        except KeyError as ex:
            logger.error("Can't process resource %r metadata: %s", res_cls, ex)

    return dict(tasks=tasks, resources=resources)
