# coding: utf-8

import logging

from sandbox import common

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sandboxsdk import environments


class ArcadiaHgRepository(sdk2.Resource):
    """ Copy of Arcadia HG repository """

    share = True
    repository = sdk2.parameters.String("Original repository url")
    rev = sdk2.parameters.String("Revision hash")
    full_size = sdk2.parameters.String("Size of unpacked repository")


class BundleArcadiaHgRepository(sdk2.ServiceTask):
    """ This task is used to create a fresh bundle of Arcadia HG repository """

    class Requirements(sdk2.Requirements):
        disk_space = 200000
        # client_tags = (
        #     ctc.Tag.STORAGE | ctc.Tag.CUSTOM_REPLICATION
        # )

    class Parameters(sdk2.Parameters):
        description = "Bundle Arcadia Hg repository"
        kill_timeout = 3600 * 5
        url = sdk2.parameters.String("Hg repository URL", default="https://arcadia-hg.yandex-team.ru/arcadia.hg")
        rev = sdk2.parameters.String("Revision", default="default")
        force_backup = sdk2.parameters.Bool("Force backup", default=True)

    def full_checkout(self, rev):
        cmd = ["hg", "clone", self.Parameters.url, "arcadia.hg"]
        with sdk2.helpers.ProcessLog(self, logger="hg_clone") as pl:
            sp.Popen(cmd, stdout=pl.stdout, stderr=sp.STDOUT).wait()
        return "arcadia.hg"

    def on_execute(self):
        environments.SandboxHgEnvironment().prepare()

        try:
            cache_dir = self.agentr.arcadia_hg_clone(self.Parameters.rev)
        except Exception:
            logging.exception("arcadia_hg_clone failed. Doing manual checkout")
            cache_dir = self.full_checkout(self.Parameters.rev)

        bundle_name = "arcadia.hg.tar.gz"
        cur_rev = sp.check_output(["hg", "parent", "--template", "{node}"], cwd=cache_dir)
        logging.info("bundling up cache at %s", cache_dir)
        cmd = ["tar", "-C", cache_dir, "-zcf", bundle_name, "."]
        with sdk2.helpers.ProcessLog(self, logger="tar_gz") as pl:
            sp.Popen(cmd, stdout=pl.stdout, stderr=sp.STDOUT).wait()

        full_size = common.fs.get_dir_size(cache_dir) * 1024

        res = ArcadiaHgRepository(
            self, "Arcadia Hg repository", bundle_name,
            rev=cur_rev, repository=self.Parameters.url,
            full_size=full_size
        )

        logging.info("pre-sharing %s", bundle_name)
        common.share.skynet_share(self.path(), bundle_name)
        logging.info("done sharing %s", bundle_name)
        sdk2.ResourceData(res).ready()

        if self.Parameters.force_backup:
            backup_id = self.server.resource[res.id].backup().get('id')
            self.Context.backup_id = backup_id
            # TODO: wait for it
            # raise sdk2.WaitTask ...
