import os
import logging

from sandbox import sdk2
from sandbox import common

from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.constants import constants as arcadiasdk_constants
from sandbox.projects.common.vcs import arc
from sandbox.projects.common import decorators


def embed_revision(init_module, rev):
    init_module = sdk2.path.Path(init_module)
    logging.info("Embedding revision %r into file %r", rev, init_module)

    if init_module.exists():
        with init_module.open() as f:
            lines = f.readlines()
    else:
        lines = ["__revision__ = None\n"]

    with init_module.open("w") as f:
        written = False
        for line in lines:
            if line.startswith("__revision__"):
                f.write(u"__revision__ = {}\n".format(rev))
                written = True
            else:
                f.write(line)
        if not written:
            f.write(u"\n__revision__ = {}\n".format(rev))


def is_arc_url(url):
    return url.startswith(sdk2.vcs.svn.Arcadia.ARCADIA_ARC_SCHEME)


def get_binary_name(mounted_path, target, results_dir=""):
    """Add a binary path to the target, assuming it's not set"""
    if common.system.inside_the_binary():
        path = os.path.join(mounted_path, target)
        ya_make_path = os.path.join(path, "ya.make")
        if os.path.exists(ya_make_path):
            from yalibrary import makelists

            make_list = makelists.from_file(ya_make_path)
            sandbox_targets = ("SANDBOX_TASK", "SANDBOX_PY3_TASK", "SANDBOX_PY23_TASK", "PY2_PROGRAM", "PY3_PROGRAM")
            sandbox_task = make_list.project.name
            if sandbox_task in sandbox_targets and len(make_list.project.get_values()) > 0:
                binary_name = make_list.project.get_values()[0].project_name
                logging.debug("Found binary name: %s", binary_name)
                return os.path.join(results_dir, target, binary_name)
            logging.info("Can't get a name of a binary from ya.make %s, make_list:\n%s", ya_make_path, make_list.dump())
            # here we assume that the binary name is the same as the top-level directory name
            return os.path.join(results_dir, target, target.rsplit("/", 1)[-1])
    else:
        executables = []
        path = os.path.join(results_dir, target)
        for file in sdk2.paths.list_dir(path, files_only=True):
            if os.access(os.path.join(path, file), os.X_OK):
                logging.debug('Found "%s" executable file', file)
                executables.append(file)
        if len(executables) == 1:
            return os.path.join(path, executables[0])

    raise common.errors.TaskError(
        "{} executable file found in specified directory '{}'. Please specify expected binary name".format(
            "Multiple ({})".format(executables) if len(executables) else "No one", target
        )
    )


def trim_target_binary(mounted_path, targets):
    """Removes binary names from each target if them are listed"""
    fixed_targets = []
    for target in targets:
        expected_dir_path = os.path.join(mounted_path, target)
        if os.path.exists(expected_dir_path):
            fixed_targets.append(target)
        else:
            path_without_binary = expected_dir_path.rsplit("/", 1)[0]
            if os.path.exists(path_without_binary):
                fixed_targets.append(target.rsplit("/", 1)[0])
            else:
                raise common.errors.TaskFailure("Target path should be either a binary file, or a directory with ya.make")

    return fixed_targets


@decorators.retries(max_tries=3, exceptions=arc.ArcNoSuchRevision)
def build(
    trunk_url, targets, build_system=arcadiasdk_constants.SEMI_DISTBUILD_BUILD_SYSTEM,
    use_yt_cache=False, arc_oauth_token=None, return_binaries=False, **build_options
):
    """
    Build something with arc, retrying for a couple of times until the desired revision is available.

    This function accepts all[1] parameters accepted by `projects.common.arcadia.sdk.do_build()` --
    refer to its docstring whenever possible.

        [1] All except these related to YT cache usage

    :param trunk_url: Arcadia trunk URL (you may also specify revision)
    :param targets: a single target, or an iterable of targets
    :param build_system: one of `projects.common.constants.*_BUILD_SYSTEM`
    :param use_yt_cache: if possible, download already built artifacts from YT cache.
        Details: https://wiki.yandex-team.ru/users/deshevoy/yt-cache
        You may also use `yt_store_pararms=arcadiasdk.YtStoreParams()` directly
    :param return_binaries: returns a list of binary names, each name for each target
    :param build_options: everything you want to pass into `sdk.do_build()`
    :return: (return code of build process, revision of code built, [target list, binary names list|if return_binaries])
    :rtype: tuple
    """

    task = sdk2.Task.current

    with arcadiasdk.mount_arc_path(trunk_url, arc_oauth_token=arc_oauth_token) as aarcadia:
        patch = build_options.pop("patch", None)
        if patch:
            # It should be sufficient for patch to be applied to patch working copy.
            path_to_patch = sdk2.svn.Arcadia.apply_patch(aarcadia, patch, task.path())
            if patch.startswith("arc:"):
                patch = path_to_patch

        yt_store_params = build_options.pop("yt_store_params", None)
        if yt_store_params is None and use_yt_cache:
            try:
                yt_store_params = arcadiasdk.YtStoreParams(
                    store=True,
                    token=sdk2.Vault.data(task.owner, "yt-token")
                )
            except common.errors.VaultError as exc:
                logging.warning("YT cache is DISABLED, as you are not eligible for its usage: %s", exc)

        targets_without_binary = trim_target_binary(aarcadia, targets)

        rev = arcadiasdk.mounted_path_svnversion(aarcadia, arc_vcs=is_arc_url(trunk_url))["revision"]
        rc = arcadiasdk.do_build(
            build_system=build_system,
            source_root=aarcadia,
            targets=list(common.utils.chain(targets_without_binary)),
            yt_store_params=yt_store_params,
            patch=patch,
            **build_options
        )

        if return_binaries:
            return rc, rev, targets, [
                get_binary_name(aarcadia, t, build_options["results_dir"]) for t in targets_without_binary
            ]
        return rc, rev


class ReleaseTTL(sdk2.parameters.String):
    @classmethod
    def cast(cls, value):
        if value == "inf":
            return super(ReleaseTTL, cls).cast(value)
        else:
            return super(ReleaseTTL, cls).cast(str(int(value)))
