from __future__ import absolute_import, print_function
from six.moves import queue
import logging
import threading
import collections

logger = logging.getLogger(__name__)


class SimpleComputationGraph(object):
    """
    Class for computation of simple job graph.

    Each job executes in separate thread, so the number of jobs shouldn't be very big.
    """

    Job = collections.namedtuple('Job', ['id', 'name', 'method', 'args', 'kwargs', 'wait_jobs'])

    def __init__(self, log=None):
        self._back_queue = queue.Queue()
        self._workers = []
        self._logger = log or logger

    def add_job(self, method, args=(), kwargs=None, wait_jobs=()):
        """
        Add job to execute. Waited jobs have to be added already.

        :param method: callable to run in job
        :type method: function
        :param args: args for `method`
        :param kwargs: kwargs for `method`
        :param wait_jobs: list if job ids to wait
        :return: job id
        """
        kwargs = kwargs or {}
        job_id = len(self._workers)
        if not all(0 <= wait_job_id < job_id for wait_job_id in wait_jobs):
            raise ValueError("It's allowed to wait only jobs that are registered already")
        job_name = '<{}|{}>'.format(job_id, method.__name__)
        job = self.Job(id=job_id, name=job_name, method=method, args=args, kwargs=kwargs, wait_jobs=wait_jobs)
        self._workers.append(threading.Thread(target=self._thread_error_reporter, args=(job, self._back_queue)))
        return job_id

    def run(self):
        for worker in self._workers:
            worker.start()
        map(threading.Thread.join, self._workers)
        rs = [self._back_queue.get() for _ in self._workers]
        if any(rs):
            raise next(iter(filter(None, rs)))

    def _thread_error_reporter(self, job, back_queue):
        if job.wait_jobs:
            self._logger.debug('Job %s: wait jobs %r', job.name, job.wait_jobs)
            for job_id in job.wait_jobs:
                self._workers[job_id].join()
        self._logger.debug('Job %s: start', job.name)
        try:
            job.method(*job.args, **job.kwargs)
            back_queue.put(None)
            self._logger.debug("Job %s: finish", job.name)
        except Exception as ex:
            self._logger.exception("Job %s: error in thread", job.name)
            back_queue.put(ex)
