# coding: utf-8

import os
import re
import shutil
import logging
import collections

from sandbox.common import enum
from sandbox.common import errors
from sandbox.common.types import misc as ctm
from sandbox.common.types import client as ctc

from sandbox import sdk2
from sandbox.sdk2 import copy as sdk_copy

from sandbox.projects.common.nanny import nanny


class Protocol(enum.Enum):
    enum.Enum.lower_case()

    RCP = None
    SVN = None
    SCP = None
    HTTP = None
    RSYNC = None
    SKYNET = None


SKYNET_FETCHING_POSTFIX = "_fetched_resource"

MetaInfo = collections.namedtuple(
    "MetaInfo",
    "protocol src_path description resource_name resource_type rsync_copy_links attributes id resource_arch"
)


class CreateResource(sdk2.Task, nanny.ReleaseToNannyTask2):
    """Creates resource from file/torrent """

    class Requirement(sdk2.Requirements):
        ram = 512
        disk_space = 4096
        client_tags = ctc.Tag.GENERIC
        cores = 1

    class Parameters(sdk2.Parameters):
        description = "Creates resource from file or torrent"
        max_restarts = 4
        kill_timeout = 1 * 60 * 60

        ipv4_required = sdk2.parameters.Bool("IPv4 network is required for resource copying")
        create_multiple_resources = sdk2.parameters.Bool(
            "Create multiple resources via JSON meta",
            default=False
        )

        with create_multiple_resources.value[False]:
            resource_type = sdk2.parameters.String("Type of resource")
            resource_name = sdk2.parameters.String("Name of resource")
            src_path = sdk2.parameters.String("Path to resource")

            with sdk2.parameters.RadioGroup("File protocol") as protocol:
                protocol.values.rcp = Protocol.RCP
                protocol.values.skynet = protocol.Value(Protocol.SKYNET, default=True)
                protocol.values.http = Protocol.HTTP
                protocol.values.svn = Protocol.SVN
                protocol.values.scp = Protocol.SCP
                protocol.values.rsync = Protocol.RSYNC

            with protocol.value[Protocol.RSYNC]:
                rsync_copy_links = sdk2.parameters.Bool(
                    "Rsync transform symlink into referent file/dir"
                )

            with sdk2.parameters.String("Resource architecture") as resource_arch:
                resource_arch.values.any = resource_arch.Value(ctm.OSFamily.ANY, default=True)
                resource_arch.values.osx = resource_arch.Value(ctm.OSFamily.OSX)
                resource_arch.values.linux = resource_arch.Value(ctm.OSFamily.LINUX)
                resource_arch.values.win_nt = resource_arch.Value(ctm.OSFamily.WIN_NT)

            resource_attrs = sdk2.parameters.Dict(
                "Extending attributes",
                description="Custom attributes to be set for resource",
            )

        with create_multiple_resources.value[True]:
            multi_config = sdk2.parameters.JSON(
                "Extension to describe multiple resources via solid JSON data",
                description="Description for Resource [multiple resources]"
            )

        with sdk2.parameters.Output:
            resources_created = sdk2.parameters.List("IDs of created resources")
            resources_config = sdk2.parameters.JSON("Aggregated parameters about created resources")
            resources_failed = sdk2.parameters.List("IDs of failed resources")

    class Context(sdk2.Context):
        meta_to_perform = None

    def on_enqueue(self):
        super(CreateResource, self).on_enqueue()
        for meta_dict in self.Context.meta_to_perform:
            MetaInfo(**meta_dict)

    def _validate_resource_meta(self,  resource_meta):
        resource_type = resource_meta["resource_type"]
        if resource_type and resource_type not in sdk2.Resource:
            logging.error("Unhandled type of resource: '%s'", resource_meta["resource_type"])
            return False

        if (
            resource_meta["src_path"].startswith("rbtorrent:") and
            resource_meta["protocol"] != Protocol.SKYNET
        ):
            logging.error("Source path contains rbtorrent, but protocol is not skynet")
            return False
        return True

    def _set_meta_defaults(self, resource_meta):
        resource_meta.setdefault("protocol", Protocol.SKYNET)
        resource_meta.setdefault("src_path", "")
        resource_meta.setdefault("rsync_copy_links", False)
        resource_meta.setdefault("attributes", dict())
        resource_meta.setdefault("description", self.Parameters.description)
        resource_meta.setdefault("id", None)
        resource_meta.setdefault("resource_arch", ctm.OSFamily.ANY)
        return resource_meta

    def _multi_config_is_valid(self):
        return (
            isinstance(self.Parameters.multi_config, list) and
            all(isinstance(item, dict) for item in self.Parameters.multi_config)
        )

    def _craft_meta(self):
        if self.Parameters.create_multiple_resources:
            metas = self.Parameters.multi_config if self._multi_config_is_valid() else []
        else:
            metas = [
                dict(
                    protocol=self.Parameters.protocol,
                    src_path=self.Parameters.src_path,
                    resource_type=self.Parameters.resource_type,
                    resource_name=self.Parameters.resource_name,
                    rsync_copy_links=self.Parameters.rsync_copy_links,
                    attributes=self.Parameters.resource_attrs,
                ),
            ]

        verified_metas = [self._set_meta_defaults(_) for _ in metas if self._validate_resource_meta(_)]
        if len(verified_metas) != len(metas):
            raise errors.TaskFailure("Some resources have invalid meta")
        return verified_metas

    def on_save(self):
        super(CreateResource, self).on_save()
        self.Context.meta_to_perform = self._craft_meta()
        self.Requirements.dns = ctm.DnsType.DNS64 if self.Parameters.ipv4_required else ctm.DnsType.DEFAULT

    @staticmethod
    def _skynet_post(dest_path, resource_abs_path, resource_meta):
        copied_files = os.listdir(dest_path)
        if not copied_files:
            raise ValueError(
                "Error while copying from SkyNet: {}".format(copied_files),
            )
        if len(copied_files) > 1:
            shutil.move(
                dest_path,
                os.path.join(os.path.dirname(dest_path), resource_meta.resource_name)
            )
        else:
            res_file = os.path.join(dest_path, copied_files[0])
            if os.path.isdir(res_file):
                os.rename(res_file, resource_abs_path)
            else:
                shutil.move(res_file, resource_abs_path)
            shutil.rmtree(dest_path)

    def _get_copier_args(self, resource_meta, resource_abs_path):
        kws = {}
        skynet_in_use = False
        remote_file_name = resource_meta.src_path
        error_message = "Corrupted protocol settings for {!r}".format(resource_meta.protocol)

        if resource_meta.protocol == Protocol.SKYNET:
            if not remote_file_name.startswith("rbtorrent:"):
                raise ValueError("Skynet Copier link format error: rbtorrent not found")
            skynet_in_use = True
        elif resource_meta.protocol == Protocol.RSYNC:
            if not remote_file_name.startswith("rsync:"):
                match = re.match(r"(.+)::(.+)", remote_file_name)
                if match:
                    remote_file_name = "rsync://{}/{}".format(match.group(1), match.group(2))
                else:
                    raise ValueError(error_message)
            if resource_meta.rsync_copy_links:
                kws = {"copy-links": True}

        if skynet_in_use:
            dest_path = os.path.join(os.path.dirname(resource_abs_path), SKYNET_FETCHING_POSTFIX)
            kws = {"fallback_to_bb": True}
        else:
            dest_path = resource_abs_path

        return skynet_in_use, dest_path, remote_file_name, kws

    def _process_resource(self, resource_meta):
        import api.copier.errors

        data = self.agentr.resource_register(
            resource_meta.resource_name,
            resource_meta.resource_type,
            resource_meta.description,
            resource_meta.resource_arch,
            resource_meta.attributes,
        )
        resource = sdk2.Resource.restore(data)

        resource_data = sdk2.ResourceData(resource)
        resource_abs_path = str(resource_data.path)

        try:
            skynet_in_use, dest_path, remote_file_name, kws = self._get_copier_args(resource_meta, resource_abs_path)

            kws["timeout"] = int(self.Parameters.kill_timeout)
            remote_copier = sdk_copy.RemoteCopy(
                remote_file_name,
                dest_path,
                log_dir=self.log_path(),
            )
            logging.info("Using %r to fetch remote data.", remote_copier)
            remote_copier(**kws)

        except api.copier.errors.ResourceNotAvailable as ex:
            raise errors.InvalidResource(ex)

        if skynet_in_use:
            self._skynet_post(dest_path, resource_abs_path, resource_meta)

    def on_execute(self):
        resources_config = []
        failed_resource_ids = []
        created_resource_ids = []
        for resource_dict in self.Context.meta_to_perform:
            resource_meta = MetaInfo(**resource_dict)
            try:
                self._process_resource(resource_meta)
                created_resource_ids.append(resource_meta.id)
                resources_config.append(resource_dict)
            except Exception as error:
                logging.exception("There are some problems during coping %s: %s", resource_meta.id, error)
                failed_resource_ids.append(resource_meta.id)

        self.Parameters.resources_config = resources_config
        self.Parameters.resources_created = created_resource_ids
        self.Parameters.resources_failed = failed_resource_ids

        if failed_resource_ids:
            logging.error("Failed resources ids: %r", failed_resource_ids)
            raise errors.TaskError(
                "Some resources are failed to copy: {!r}".format(failed_resource_ids)
            )
