# -*- coding: utf-8 -*-
import os
import json
import shutil
import logging
import textwrap
import platform
import multiprocessing as mp

from sandbox import sdk2
from sandbox.sdk2.helpers.process import subprocess

from sandbox.common import rest
from sandbox.common import enum
from sandbox.common import utils
from sandbox.common import config
from sandbox.common import errors
from sandbox.common import patterns
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr

import sandbox.projects.sandbox.common
from sandbox.projects.common import binary_task
from sandbox.projects.common import task_env
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.common.constants import constants as arcadia_sdk_constants

logger = logging.getLogger(__name__)


class TargetPlatforms(enum.Enum):
    """Available ya.make target platforms for cross-compilation"""

    enum.Enum.lower_case()

    LINUX = None
    DARWIN = None
    WINDOWS = None
    DARWIN_ARM64 = "default-darwin-arm64"


WINDOWS_BIN_EXT = ".exe"
ARC_TOKEN_NAME = "arc-token"


# FIXME: put this into sdk2.helpers.resource later on
def binary_resource_attrs(binary_path, logger=logging):  # type: (str, logging.Logger) -> dict
    p = subprocess.Popen(
        [str(binary_path), "content", "--self-info"],
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
        env={"Y_PYTHON_ENTRY_POINT": "sandbox.taskbox.binary:main"},
    )
    out, err = p.communicate()
    logger.debug("Binary '%s' exited with code %s. STDOUT:\n%s", binary_path, p.returncode, out)
    if p.returncode != 0 or err:
        logger.warning("STDERR\n:%s", err)
    return json.loads(out)["resource_attributes"]


class ComplexJSON(sdk2.parameters.JSON):
    @classmethod
    def cast(cls, value):
        v = [value] if isinstance(value, dict) else value
        if not (
            isinstance(v, list) and
            all(isinstance(data, dict) for data in v)
        ):
            raise ValueError("A dict, or a list of dicts, is required")

        return super(ComplexJSON, cls).cast(value)


class DeployBinaryTask(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """Build and verify resource with binary task runner"""

    MINIMAL_RAMDRIVE_SIZE = 30 * 1024
    PREPROD_TOKEN_NAME = "preprod-oauth-token"
    PREPROD_API_URL = "https://www-sandbox1.n.yandex-team.ru/api/v1.0"
    TARGET_PLATFORM_TO_SANDBOX = {
        TargetPlatforms.WINDOWS: ctm.OSFamily.WIN_NT,
        TargetPlatforms.LINUX: ctm.OSFamily.LINUX,
        TargetPlatforms.DARWIN: ctm.OSFamily.OSX,
        TargetPlatforms.DARWIN_ARM64: ctm.OSFamily.OSX_ARM,
    }

    class Parameters(sdk2.Parameters):

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia URL; must always point to Arcadia root (branches/tags are allowed). If empty, trunk URL is used."
        )
        target = sdk2.parameters.String(
            "Path to target task relative to Arcadia root",
            description=(
                "Example: sandbox/projects/sandbox/<binary_name>"
                "(if binary program's custom name is not set, it's being detected automatically)"
            ),
            required=True,
        )
        arcadia_patch = build_parameters.ArcadiaPatch(group=None, default="")
        attrs = sdk2.parameters.Dict("Binary task archive attributes")
        check_types = sdk2.parameters.List("Check these task types are present in result program")

        integrational_check = sdk2.parameters.Bool(
            "Run custom tasks on built binary and wait for their completion",
            description=(
                "These tasks will serve as integrational tests: the parent task will wait until "
                "all children have stopped executing, be it failure or success, and then check their statuses"
            )
        )
        with integrational_check.value[True]:
            keep_binary_on_fail = sdk2.parameters.Bool(
                "Don't remove built binary if custom tasks fail",
                description=(
                    "By default, the binary is removed."
                    "This allows you to download it and examine instead"
                )
            )
            integrational_check_payload = ComplexJSON(
                "REST API data for task(s) creation",
                description=(
                    "Schema: https://sandbox.yandex-team.ru/api/redoc#operation/task_list_post. "
                    "To launch multiple tasks, use a list of dicts: "
                    "[{\"type\": \"MY_TASK\", ...}, {\"type\": \"ANOTHER_TASK\", ...}]"
                ),
                default={
                    "type": "TEST_TASK_2",
                },
            )

        arc_oauth_token = sdk2.parameters.YavSecret(
            "Yav secret with OAuth token for Arc. '{}' is used as a key by default.".format(ARC_TOKEN_NAME),
            required=True,
        )

        with sdk2.parameters.Group("Build options") as build_options:
            build_system = build_parameters.BuildSystem(
                group=None, default=arcadia_sdk_constants.SEMI_DISTBUILD_BUILD_SYSTEM
            )
            target_platform = build_parameters.TargetPlatformFlags(
                default=TargetPlatforms.LINUX,
                choices=[(p, p) for p in TargetPlatforms]
            )
            extra_build_arguments = sdk2.parameters.List(
                "Additional arguments that will be passed to the ya make build"
            )

            with sdk2.parameters.List(
                "If there is at least one item task will create directory with binaries for every platform"
            ) as package_target_platforms:
                for p in TargetPlatforms:
                    setattr(package_target_platforms.values, p, p)

            ignore_recurses = build_parameters.IgnoreRecurses(default=True, group=None)
            ramdrive_for_cache = sdk2.parameters.Bool("Speed up build by storing cache in TMPFS", default=True)
            use_yt_cache = sdk2.parameters.Bool("When possible, download artifacts from YT cache", default=True)
            with use_yt_cache.value[True]:
                yt_token_vault = sdk2.parameters.Vault(
                    "Vault YT token vault entry (owner:name, or only name)", default="yt-token"
                )
                yt_token_yav = sdk2.parameters.YavSecretWithKey(
                    "Yav secret with YT token (priority over the Vault secret)", default=""
                )

        with sdk2.parameters.Group("Test options") as test_options:
            test = build_parameters.TestParameter(group=None, default=False)
            with test.value[True]:
                test_params = build_parameters.TestCustomParameters(group=None, default="")
                test_filters = build_parameters.TestFilters(group=None, default="")
                test_size_filter = build_parameters.TestSizeFilter(group=None, default="")

        with sdk2.parameters.Group("Release options") as release_options:
            release_ttl = sandbox.projects.sandbox.common.ReleaseTTL("TTL for released resources", default="inf")

        resource_for_parent = sdk2.parameters.Bool("Create resource for parent task", ui=None)

        ext_params = binary_task.LastBinaryReleaseParameters()

        auto_deploy_to_preproduction = sdk2.parameters.Bool(
            "Push resources to preproduction", default=False, do_not_copy=True
        )

        with sdk2.parameters.RadioGroup(
            "Whether to set 'taskbox_enabled' parameter for generated resource"
        ) as taskbox_enabled:
            taskbox_enabled.values["auto"] = taskbox_enabled.Value("Auto", default=True)
            taskbox_enabled.values["on"] = taskbox_enabled.Value("On")
            taskbox_enabled.values["off"] = taskbox_enabled.Value("Off")

    class Requirements(task_env.BuildRequirements):
        ram = 30 * 1024
        disk_space = 10 * 1024   # 10 Gb

    class Context(sdk2.Context):
        children = []
        experimental_binary_id = None

    def __get_children(self):
        tasks = self.server.task.read(
            id=",".join(map(str, self.Context.children)),
            limit=len(self.Context.children)
        )
        return tasks

    def __delete_experimental_binary(self):
        self.server.batch.resources["delete"].update(
            id=[self.Context.experimental_binary_id],
            comment="Remove unnecessary experimental binary"
        )

    def __arc_token(self):
        if not self.Parameters.arc_oauth_token:
            return None
        key = self.Parameters.arc_oauth_token.default_key or ARC_TOKEN_NAME
        return self.Parameters.arc_oauth_token.data().get(key, None)

    def __update_from_task_params(self, attrs, is_experimental=False):
        if self.Parameters.taskbox_enabled == "auto":
            attrs[ctr.BinaryAttributes.TASKBOX_ENABLED] = (
                self.Parameters.target_platform == TargetPlatforms.LINUX
                and not self.Parameters.target.startswith("sandbox/projects/")
            )
            logging.debug(
                "Define 'taskbox_enabled' for resource as '%s': target platform %s, target path '%s'",
                attrs[ctr.BinaryAttributes.TASKBOX_ENABLED],
                self.Parameters.target_platform,
                self.Parameters.target,
            )
        elif self.Parameters.taskbox_enabled == "on":
            attrs[ctr.BinaryAttributes.TASKBOX_ENABLED] = True
        else:
            attrs[ctr.BinaryAttributes.TASKBOX_ENABLED] = False

        # if generic attrs contains own value for `taskbox_enabled`, it has more priority and overwrites
        if is_experimental:
            val = self.Parameters.attrs.get(ctr.BinaryAttributes.TASKBOX_ENABLED, None)
            if val:
                attrs[ctr.BinaryAttributes.TASKBOX_ENABLED] = val
        else:
            attrs.update(**self.Parameters.attrs)

        attrs.setdefault("sync_upload_to_mds", False)

        return attrs

    @property
    def binary_executor_query(self):
        query = super(DeployBinaryTask, self).binary_executor_query
        query.update({"owner": "SANDBOX"})
        return query

    def platform_to_sandbox(self, pl):
        return self.TARGET_PLATFORM_TO_SANDBOX[pl]

    @patterns.singleton_property
    def resource_arch(self):
        return self.TARGET_PLATFORM_TO_SANDBOX.get(self.Parameters.target_platform, ctm.OSFamily.ANY)

    @sdk2.header()
    def integrational_checks_status(self):
        if not self.Parameters.integrational_check:
            return None

        def wrap_task(t):
            task_info = "{type} #<a href='{url}'>{id}</a>".format(
                type=t["type"], url=utils.get_task_link(t["id"]), id=t["id"]
            )
            coloring = "status_success" if t["status"] in ctt.Status.Group.SUCCEED else "status_exception"
            return textwrap.dedent("""
                <tr>
                  <td style="padding: 5px">{task_info}</td>
                  <td style="padding: 5px"><span class="status {coloring}">{status}</span></td>
                </tr>
            """.format(
                task_info=task_info, coloring=coloring, status=t["status"]
            ))

        return textwrap.dedent("""
            <table>
              <tbody>
                <tr>
                  <th style="padding: 5px">Task</th>
                  <th style="padding: 5px">Status</th>
                </tr>
                {tasks}
              </tbody>
            </table>
        """.format(
            tasks="\n".join(wrap_task(task) for task in self.__get_children()["items"])
        ))

    @patterns.singleton_property
    def build_cache_dir(self):
        if self.Parameters.ramdrive_for_cache:
            path = self.ramdrive.path / "build_cache_dir"
            path.mkdir(exist_ok=True)
            return str(path)
        return None

    def on_save(self):
        super(DeployBinaryTask, self).on_save()
        if self.Parameters.ramdrive_for_cache:
            tmpfs_size = max(
                0 if self.Requirements.ramdrive is None else self.Requirements.ramdrive.size,
                self.MINIMAL_RAMDRIVE_SIZE
            )
            self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, tmpfs_size, None)

    def on_enqueue(self):
        with self.memoize_stage.check_once:
            if self.Parameters.integrational_check:
                data = self.Parameters.integrational_check_payload
                if isinstance(data, dict):
                    data = [data]
                data = list(filter(None, data))
                if not data:
                    raise errors.TaskFailure("No empty payload allowed")
                self.Parameters.integrational_check_payload = data

    @staticmethod
    def _to_ya_make_platform(platform):
        platform = platform.lower()
        if TargetPlatforms.DARWIN in platform:
            return TargetPlatforms.DARWIN
        elif TargetPlatforms.WINDOWS in platform:
            return TargetPlatforms.WINDOWS
        return TargetPlatforms.LINUX

    def build_and_test(self):
        results_dir = str(self.path("build_result"))

        test_size_filter = filter(
            None,
            map(
                lambda s: s.strip(),
                self.Parameters.test_size_filter.split(",")
            )
        )

        yt_store_params = None
        if self.Parameters.use_yt_cache:
            if self.Parameters.yt_token_yav is None:
                record = self.Parameters.yt_token_vault
                if record.owner is None:
                    record.owner = self.owner
                yt_store_params = arcadia_sdk.YtStoreParams(store=True, token=record.data())
            else:
                logging.info("Getting YT token from yav secret %s", self.Parameters.yt_token_yav)
                decoded_secret = self.Parameters.yt_token_yav
                yt_store_params = arcadia_sdk.YtStoreParams(store=True, token=decoded_secret.value())

        arc_oauth_token = self.__arc_token()

        target_platforms = set(self.Parameters.package_target_platforms)
        target_platforms.add(self.Parameters.target_platform)

        binary_resource_path = self.path("binary_resource")

        platform_to_path = {}

        logger.info("Build for platforms: %s", ",".join(target_platforms))

        binary_build_params = {
            "build_system": self.Parameters.build_system,
            "patch": self.Parameters.arcadia_patch,
            "yt_store_params": yt_store_params,
            "clear_build": False,
            "results_dir": results_dir,
            "separate_result_dirs": True,
            "test": self.Parameters.test,
            "test_params": self.Parameters.test_params or {},
            "test_filters": self.Parameters.test_filters,
            "test_size_filter": test_size_filter,
            "ignore_recurses": self.Parameters.ignore_recurses,
            "build_dir": self.build_cache_dir,
            "build_threads": mp.cpu_count() * 2 // 3,  # Decrease number of threads to avoid tests timeouts
            "ya_make_extra_parameters": self.Parameters.extra_build_arguments,
        }

        for target_platform in target_platforms:
            build_params = dict(
                [
                    ("trunk_url", self.Parameters.arcadia_url),
                    ("targets", [self.Parameters.target]),
                    ("target_platform", target_platform),
                    ("arc_oauth_token", arc_oauth_token),
                    ("return_binaries", True),
                ],
                **binary_build_params
            )
            (
                build_return_code,
                revision,
                targets,
                binaries,
            ) = sandbox.projects.sandbox.common.build(**build_params)
            # we always have only one target
            binary_path = os.path.join(results_dir, targets[0], binaries[0])
            if (
                target_platform == TargetPlatforms.WINDOWS and
                not binary_path.endswith(WINDOWS_BIN_EXT)
            ):
                binary_path = binary_path + WINDOWS_BIN_EXT
            logger.debug("Binary path is {}".format(binary_path))
            binary_path_dir = binary_resource_path.joinpath(target_platform)
            os.makedirs(str(binary_path_dir))
            new_binary_path = binary_path_dir.joinpath(os.path.basename(binary_path))
            shutil.move(str(binary_path), str(new_binary_path))
            platform_to_path[target_platform] = new_binary_path
            logger.debug("Build returned %s; target was built upon revision r%d", build_return_code, revision)

        logger.info("Result binaries: %s", platform_to_path)
        current_platform = self._to_ya_make_platform(platform.platform())
        platforms_compatible = (current_platform in platform_to_path)
        host_binary_path = str(platform_to_path.get(current_platform))

        if not platforms_compatible:
            logger.debug(
                "Current platform is not the target one, also need to build %s platform binary", current_platform
            )

            host_binary_path_suffix = "host_platform_binaries"
            binary_build_params["results_dir"] = os.path.join(results_dir, host_binary_path_suffix)

            host_build_return_code, host_revision, targets, binaries = sandbox.projects.sandbox.common.build(
                self.Parameters.arcadia_url, [self.Parameters.target],
                target_platform=current_platform,
                arc_oauth_token=arc_oauth_token,
                return_binaries=True,
                **binary_build_params
            )
            host_binary_path = os.path.join(results_dir, host_binary_path_suffix, binaries[0])
            if (
                self.Parameters.target_platform == TargetPlatforms.WINDOWS and
                host_binary_path.endswith(WINDOWS_BIN_EXT)
            ):
                host_binary_path = host_binary_path[:-len(WINDOWS_BIN_EXT)]
            logger.debug(
                "Host build returned %s; target was built upon revision r%d",
                host_build_return_code,
                host_revision
            )

        if self.Parameters.check_types:
            test_result, _ = subprocess.Popen(
                [host_binary_path, "content", "--list-types"],
                stdout=subprocess.PIPE,
                env={"Y_PYTHON_ENTRY_POINT": "sandbox.taskbox.binary:main"},
            ).communicate()
            logger.debug("Binary contains task types: %s", test_result)
            absent = set(filter(lambda v: v, self.Parameters.check_types)) - set(json.loads(test_result)["types"])
            if absent:
                raise errors.TaskFailure("Task types not present in resulting binary: {}".format(" ".join(absent)))

        # With integrational tests:
        # - Register an experimental resource and mark it ready, so that it could be used outside of the current task.
        #     We can't have a single resource, because other tasks may use it and fail while tests aren't done yet;
        # - Start test tasks using the experimental resource as binary and go to WAIT_TASK;
        # - Wake up and check children status;
        # - Re-download the resource's data (unfortunately, it has to be done) and re-register it as non-experimental.
        #
        # Without integrational tests:
        # - Finalize the resource the second it's registered.

        description = "Tasks binary for {}".format(self.Parameters.target)
        attrs = binary_resource_attrs(host_binary_path)
        if not self.Parameters.package_target_platforms:
            if not platforms_compatible:
                attrs[ctr.BinaryAttributes.BINARY_HASH] = utils.md5sum(str(new_binary_path))

        if self.Parameters.integrational_check:
            attrs = self.__update_from_task_params(attrs, True)
            description = "{} (EXPERIMENTAL)".format(description)
            attrs["released"] = ctt.ReleaseStatus.UNSTABLE
        else:
            attrs = self.__update_from_task_params(attrs)

        if not self.Parameters.package_target_platforms:
            attrs["path"] = str(new_binary_path)
        else:
            attrs["path"] = str(binary_resource_path)
            for target_platform in target_platforms:
                self.resource_system_attributes[
                    self.platform_to_sandbox(target_platform) + "_platform"
                ] = str(platform_to_path[target_platform].relative_to(binary_resource_path))
            attrs["system_attributes"] = self.resource_system_attributes
        binary = sdk2.service_resources.SandboxTasksBinary(
                    self if not self.Parameters.resource_for_parent or self.Parameters.integrational_check else self.parent,
                    description=description,
                    arch=self.resource_arch,
                    **attrs
        )
        sdk2.ResourceData(binary).ready()
        if self.Parameters.integrational_check:
            self.Context.experimental_binary_id = binary.id

    def run_integrational_check(self):
        futures = []
        with rest.Batch(self.server) as api:
            for task_data in self.Parameters.integrational_check_payload:
                description = task_data.get("description", "Integrational testing")
                requirements = task_data.get("requirements", {})
                requirements.update(
                    tasks_resource=self.Context.experimental_binary_id,
                )

                task_data.update(
                    parent=dict(id=self.id),
                    children=True,
                    description="{} (launched for task #{})".format(description, self.id),
                    owner=self.owner,
                    notifications=[],
                    requirements=requirements,
                )

                futures.append(api.task.create(**task_data))

        for future in futures:
            task = future.result()
            logging.info("Created subtask #%d of type %r for integrational testing", task["id"], task["type"])
            if task["sdk_version"] == 1:
                raise errors.TaskFailure(
                    "One of integrational testing tasks, {!r} #{}, is of SDK1 type -- "
                    "they can't run from a binary. Aborting".format(task["type"], task["id"])
                )
            self.Context.children.append(task["id"])

        reply = self.server.batch.tasks.start.update(id=self.Context.children)
        if any(_["status"] == "ERROR" for _ in reply):
            message = "Failed to start all integrational testing tasks successfully (see debug.log)"
            logging.error("%s: %r", message, reply)
            raise errors.TaskFailure(message)

        raise sdk2.WaitTask(
            self.Context.children,
            tuple(ctt.Status.Group.BREAK) + tuple(ctt.Status.Group.FINISH)
        )

    def get_integrational_check_results(self):
        tasks = self.__get_children()
        failures = [t for t in tasks["items"] if t["status"] not in ctt.Status.Group.SUCCEED]
        if failures:
            if not self.Parameters.keep_binary_on_fail:
                self.__delete_experimental_binary()

            is_singular = len(self.Context.children) == 1
            raise errors.TaskFailure(
                "{} acceptance task{} {}: {}".format(
                    "One" if is_singular else len(failures),
                    "" if is_singular else "s",
                    "has failed" if is_singular else "have failed",
                    ", ".join(str(f["id"]) for f in failures),
                )
            )

    def reregister_binary_resource(self):
        experimental_resource = sdk2.Resource[self.Context.experimental_binary_id]
        resource_data = sdk2.ResourceData(experimental_resource)
        current_platform = self.platform_to_sandbox(self._to_ya_make_platform(platform.platform()))
        binary_path = str(resource_data.path)
        if self.Parameters.package_target_platforms:
            host_bin_path = getattr(experimental_resource.system_attributes, current_platform + "_platform")
            binary_path = os.path.join(binary_path, host_bin_path)
        attrs = binary_resource_attrs(binary_path)
        attrs = self.__update_from_task_params(attrs)

        duplicate_dir = self.path("tested-binary")
        if self.Parameters.package_target_platforms:
            shutil.copytree(str(resource_data.path), str(duplicate_dir))
            path = duplicate_dir
            target_platforms = set(self.Parameters.package_target_platforms)
            target_platforms.add(current_platform)
            resource_system_attributes = {}
            for target_platform in target_platforms:
                current_platform = self.platform_to_sandbox(target_platform) + "_platform"
                resource_system_attributes[current_platform] = getattr(experimental_resource.system_attributes, current_platform)
            attrs["system_attributes"] = resource_system_attributes
        else:
            duplicate_dir.mkdir()
            shutil.copy(binary_path, str(duplicate_dir))
            path = duplicate_dir / os.path.basename(binary_path)

        sdk2.ResourceData(sdk2.service_resources.SandboxTasksBinary(
            self if not self.Parameters.resource_for_parent else self.parent,
            description="Tasks binary for {}".format(self.Parameters.target),
            path=path,
            arch=self.resource_arch,
            **attrs
        )).ready()
        self.__delete_experimental_binary()

    def push_to_preprod(self):
        try:
            preprod_token = sdk2.Vault.data(self.owner, self.PREPROD_TOKEN_NAME)
        except errors.VaultError:
            logging.error("Failed to get preprod OAuth token", exc_info=True)
            return

        api = rest.Client(self.PREPROD_API_URL, preprod_token)

        res = sdk2.service_resources.SandboxTasksBinary.find(task=self).first()
        attributes = dict(res)
        attributes.pop("backup_task", None)

        task = api.task(
            type="REMOTE_COPY_RESOURCE_2",
            owner=self.owner,
            description="Sandbox binary task resource obtained by <a href='{}'>#{}</a>".format(
                utils.get_task_link(self.id),
                self.id,
            ),
            custom_fields=[
                {
                    "name": "resource_type",
                    "value": str(res.type),
                },
                {
                    "name": "created_resource_name",
                    "value": str(res.path),
                },
                {
                    "name": "remote_file_name",
                    "value": res.skynet_id,
                },
                {
                    "name": "resource_attrs",
                    "value": ",".join("{}={}".format(key, value) for key, value in attributes.iteritems()),
                },
                {
                    "name": "system_resource_attrs",
                    "value": ",".join(
                        "{}={}".format(key, value) for key, value in self.resource_system_attributes.iteritems()
                    ),
                },
                {
                    "name": "resource_arch",
                    "value": self.resource_arch
                }
            ],
            kill_timeout=5 * 60
        )["id"]
        self.Context.preproduction_task_id = task
        api.batch.tasks.start = {
            "comment": "Actualize tasks code on preprod",
            "id": [task]
        }
        logging.info("Task #%r enqueued on preprod", task)

    def on_execute(self):
        self.resource_system_attributes = {}
        super(DeployBinaryTask, self).on_execute()
        with self.memoize_stage.build_and_test(commit_on_entrance=False):
            self.build_and_test()

        if self.Parameters.integrational_check:
            with self.memoize_stage.run_subtasks(commit_on_entrance=False):
                self.run_integrational_check()

            with self.memoize_stage.verify_subtasks:
                self.get_integrational_check_results()
                self.reregister_binary_resource()

        if self.Parameters.auto_deploy_to_preproduction and self.owner == config.Registry().common.service_group:
            self.push_to_preprod()

    def on_release(self, parameters_):
        """
        Task specific actions. Executed when release has submitted.

        :param parameters_: release specific parameters
        """
        logging.debug("Release parameters: %r", parameters_)
        self._send_release_info_to_email(parameters_)
        self.mark_released_resources(parameters_["release_status"], ttl=self.Parameters.release_ttl or "inf")
