# coding: utf-8

import time
import random
import logging
import operator

from sandbox import common
from sandbox import sdk2
import sandbox.common.types.task as ctt

from sandbox.projects.common import utils
from sandbox.projects.sandbox import backup_resource_2


class EnsureResourcesRedundancy(sdk2.ServiceTask):
    """
    Ensures resources redundancy on the storage host, where the task will run on.
    It will create (a number of, depending on chunk's size limit) a sub-task, which will
    create a copy of each resource, which exists on the host, but has no copy on any host from
    `H@SANDBOX_STORAGE` hosts group.

    Note:   The task currently does not take into account `auto_backup` property on the resource's class.
            This check will be implemented in future (at the time, the task will be ready to run at any
            Sandbox host, not only on these belonging to `H@SANDBOX_STORAGE` hosts group).
    """

    class Requirements(sdk2.Requirements):
        disk_space = 10
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Task.Context):
        children = None

    class Parameters(sdk2.Task.Parameters):
        @common.utils.classproperty
        def host2replicate_choices(self):
            return [(h, h) for h in sorted(_["id"] for _ in common.rest.Client().client[:100500]["items"])]

        host2replicate = sdk2.parameters.String("Host ID to replicate", required=True)
        host2replicate.choices = host2replicate_choices

        copy_chunk_size_limit = sdk2.parameters.Integer(
            "Chunk size in GiB of resources to be backed up in a single task",
            required=True, default_value=300,
            description="Limits total size in GiB of resources to be backed up in a single task. "
                        "The limit does not guarantee the total size of the list will be exactly of "
                        "the size specified, it guarantees only that the total size will be not less than"
                        " the limit specified, if there's enough resources to return."
        )

        copy_total_size_limit = sdk2.parameters.Integer(
            "Total size in GiB of resources to be backed up in a single task",
            required=True, default_value=-1,
            description="Limits total size in GiB of resources to be backed up. Negative value means no limit. "
                        "The limit does not guarantee the total size of the list will be exactly of "
                        "the size specified, it guarantees only that the total size will be not less than "
                        "the limit specified, if there's enough resources to return."
        )

        with sdk2.parameters.String("Priority class for children tasks", required=True) as child_priority_class:
            for value in sorted(map(operator.itemgetter(0), ctt.Priority.Class.iteritems())):
                child_priority_class.values[value] = child_priority_class.Value(value, default=(value == "BACKGROUND"))

        with sdk2.parameters.String("Priority subclass for children tasks", required=True) as child_priority_subclass:
            for value in sorted(map(operator.itemgetter(0), ctt.Priority.Subclass.iteritems())):
                child_priority_subclass.values[value] = child_priority_subclass.Value(value, default=(value == "LOW"))

        target_host = sdk2.parameters.String(
            "Backup destination", description="Host, on which created `BACKUP_RESOURCE_2` tasks should be scheduled."
        )
        target_host.choices = common.utils.singleton_classproperty(
            lambda _: tuple(
                (p, v)
                for p, v in common.utils.chain(
                    [("ANY", "")],
                    ((h, h) for h in sorted(common.config.Registry().server.storage_hosts))
                )
            )
        )

        with sdk2.parameters.String("DC Affinity") as dc_affinity:
            dc_affinity.other = dc_affinity.Value("Other", default=True)
            dc_affinity.same = "Same"

    def fix_insufficient_redundancy(self):
        now = time.time()
        host = self.Parameters.host2replicate
        resources_api_result = self.server.resource[host].insufficient_redundancy[:]
        resources = resources_api_result.get("items")

        random.shuffle(resources)
        logging.info(
            "Found %d resources with no copy for host '%s' in %.2fs totally for %s.",
            len(resources), host, time.time() - now, common.utils.size2str(sum([r["size"] for r in resources]))
        )

        try:
            chunk_limit = int(self.Parameters.copy_chunk_size_limit)
        except (KeyError, ValueError, TypeError):
            raise common.errors.TaskFailure("Chunk size limit is not specified. Cannot execute the task.")
        if chunk_limit < 1:
            raise common.errors.TaskFailure("Chunk size limit is negative. Cannot execute the task.")

        try:
            total_limit = int(self.Parameters.copy_total_size_limit)
        except (KeyError, ValueError, TypeError):
            raise common.errors.TaskFailure("Total size limit is not specified. Cannot execute the task.")
        priority = (self.Parameters.child_priority_class, self.Parameters.child_priority_subclass)
        target = self.Parameters.target_host

        tasks = []
        total_size = 0
        my_dc = self.server.client[host][:]["dc"]
        logging.info(
            "Own DC: %r, chunk limit: %s, total limit: %s, priority: %s, target: %s",
            my_dc, common.utils.size2str(chunk_limit << 30),
            "None" if total_limit < 0 else common.utils.size2str(total_limit << 30),
            priority, target
        )

        base_ctx = {"register_deps": False, "replication": True, "source": host}
        dc_affinity = self.Parameters.dc_affinity
        if dc_affinity == "same":
            base_ctx["dc"] = my_dc
        elif dc_affinity == "other":
            base_ctx["avoid_dc"] = my_dc

        while resources:
            if total_limit >= 0:
                left = ((total_limit << 30) - total_size) >> 30
                if left <= 0:
                    logging.info(
                        "Total size limit reached (%s <= %s).",
                        common.utils.size2str(total_limit << 30), common.utils.size2str(total_size)
                    )
                    break
                chunk_limit = min(left, chunk_limit)

            size = 0
            counter = 0
            while size < chunk_limit << 30 and counter < len(resources):
                size += resources[counter].get("size")
                counter += 1
            chunk, resources = (
                map(operator.itemgetter("id"),
                    sorted(resources[:counter], key=operator.itemgetter("size"), reverse=True)),
                resources[counter:]
            )
            description = "#{0} for task:{1} ({2}) {3} resources totally for {4}".format(
                len(tasks), self.id, self.Parameters.description, len(chunk), common.utils.size2str(size)
            )
            ctx = base_ctx.copy()
            ctx.update(
                replication=True,
                resource_id=", ".join(map(str, chunk)),
                chunk_size=size * 100 / (chunk_limit << 30),
                redundancy_semaphore=not target
            )
            task = backup_resource_2.BackupResource2(
                self,
                description=description,
                owner=self.owner,
                priority=priority,
                tags=[backup_resource_2.BackupResource2.Tag.REDUNDANCY],
                **ctx
            )
            task.Requirements.disk_space = (size >> 20) + 10
            task.Requirements.host = target or ""
            task.save()
            task.enqueue()
            total_size += size
            logging.info(
                "Sub-task #%s '%s' created (sub-total is %s). Resources to copy: %r",
                task.id, description, common.utils.size2str(total_size), chunk
            )
            tasks.append(task)

        logging.info("Totally %d tasks created.", len(tasks))
        return tasks

    def on_execute(self):
        tasks = self.Context.children
        if tasks:
            utils.check_subtasks_fails(custom_subtasks=tasks)
            return

        # First of all, create tasks for redundancy.
        with sdk2.helpers.ProgressMeter("Checking and fixing resources with insufficient redundancy."):
            tasks = self.fix_insufficient_redundancy()

        if tasks:
            self.Context.children = [t.id for t in tasks]
            raise sdk2.WaitTask(self.Context.children, tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK))
