import os
import time
import errno
import socket
import random
import logging
import inspect
import hashlib
import tarfile

from sandbox import sandboxsdk

from sandbox import common
from sandbox.common import errors
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import string


class SocketHandler(object):
    """  Socket handler for input data processing. Outside of main class to avoid pickling problems. """
    __metaclass__ = common.utils.SingletonMeta

    # Timeout on any operation on socket.
    TIMEOUT = 60

    Proto = common.upload.Proto

    def __init__(self):
        self.pid2cls = {
            _.ID: _
            for _ in self.Proto.__dict__.itervalues()
            if inspect.isclass(_) and issubclass(_, self.Proto.PacketIface) and _ is not self.Proto.PacketIface
        }

        self.fqdn = common.config.Registry().this.fqdn
        self.sock = socket.socket(socket.AF_INET6)
        self.sock.settimeout(self.TIMEOUT)
        self.sock.bind(("", 0))
        self.sock.listen(1)
        self._gen = None

    @property
    def addr(self):
        return self.fqdn, self.sock.getsockname()[1]

    def parser(self):
        peer, addr = self.sock.accept()
        peer.settimeout(self.TIMEOUT)
        logging.debug("Peer socket accepted.")

        def reliable_read(amount):
            data, chunk = "", True
            while len(data) < amount:
                for _ in range(30):
                    try:
                        chunk = peer.recv(amount - len(data))
                        if not chunk:
                            return chunk
                        data = chunk if not data else data + chunk
                        break
                    except socket.timeout:
                        return ""
                    except socket.error as ex:
                        if ex.errno != errno.EAGAIN or _ == 29:
                            raise
                        time.sleep(.1)
            return data

        greets = None
        pr = self.Proto()
        while True:
            hdr = reliable_read(pr.FMT.size)
            if not hdr:
                break
            pid, sz = pr.FMT.unpack(hdr)
            data = reliable_read(sz)
            if sz and not data:
                break
            obj = self.pid2cls[pid]().__setstate__(data)
            # logging.debug("Received a new packet %s of %d bytes length.", obj, sz)  # DEBUG: DEBUG
            if not greets:
                assert isinstance(obj, self.Proto.Greetings)
                obj.addr = addr
                greets = obj
            elif isinstance(obj, pr.Break):
                raise common.errors.TaskStop("Stream interrupted")
            yield obj
        logging.debug("Peer socket disconnected.")
        self._gen = None

    def __iter__(self):
        if not self._gen:
            self._gen = self.parser()
        for _ in self._gen:
            yield _


class TarballStreamReader(object):
    """ A helper class, which will consume data queue and will provide a data for tarball stream processor. """

    def __init__(self, task, expected, reader):
        self.task = task
        self.expected = expected
        self.reader = reader

        self.sha1 = hashlib.sha1()
        self.common_prefix = ""
        self.finished = False
        self.progress = 0
        self.received = 0
        self.buffer = ""

    def extractall(self, path):
        with tarfile.open(mode="r|", fileobj=self, bufsize=0x2FFFFF) as tar:
            tar.extractall(path, members=self.members(tar))
        self.common_prefix = self.common_prefix.strip(os.path.sep)

    def read(self, amount):
        if self.buffer and amount <= len(self.buffer):
            chunk, self.buffer = self.buffer[:amount], self.buffer[amount:]
        elif self.finished:
            chunk, self.buffer = self.buffer, ""
        else:
            try:
                pkg = self.reader.next()
            except StopIteration:
                logging.warning("Stream interrupted at %d bytes. Waiting for upload resume.", self.received)
                self.task.update_stream_status(self.received)
                sh = SocketHandler()
                self.reader = iter(sh)

                self.task.handle_greetings(self.reader.next())
                pkg = self.reader.next()

            self.received += len(pkg.data)
            assert isinstance(pkg, common.upload.Proto.DataChunk)

            curp = self.received * 100 / self.expected
            if curp != self.progress:
                logging.debug(
                    "Received so far %s (%d bytes, %s%%)",
                    common.utils.size2str(self.received), self.received, curp
                )
            self.progress = curp
            if not pkg.data:
                self.finished = True

            chunk = self.buffer + pkg.data
            if len(chunk) > amount:
                chunk, self.buffer = chunk[:amount], chunk[amount:]
            else:
                self.buffer = ""
        self.sha1.update(chunk)
        return chunk

    def members(self, members):
        for tarinfo in members:
            logging.debug("Extracting %r", tarinfo.name)
            prefix = tarinfo.name.split(os.path.sep)[0]
            if prefix == "tmp":
                raise common.errors.TaskError("'tmp' is reserved directory name, use different name")
            self.common_prefix = self.common_prefix or prefix
            if prefix != self.common_prefix:
                raise errors.TaskError(
                    "There are no common directory of uploaded files: '{}' differs with '{}'".format(
                        self.common_prefix, prefix
                    )
                )
            tarinfo.mode |= 0o444  # Allow anybody read the file
            yield tarinfo


class HTTPUpload(nanny.ReleaseToNannyTask, sandboxsdk.task.SandboxTask):
    """
    DEPRECATED. Use HTTP_UPLOAD_2 instead.

    "Backend" task for resource upload via proxy.sandbox.yandex-team.ru using HTTP protocol.
    It should **not** be used directly, without :module:`common.upload` module.
    """

    class ResourceType(sandboxsdk.parameters.SandboxStringParameter):
        name = 'resource_type'
        description = 'Resource type to be created'
        required = True

    class ResourceAttrs(sandboxsdk.parameters.SandboxParameter):
        name = 'resource_attrs'
        description = 'Resource attributes to be set on resource creation in form of dictionary'
        required = True

        @classmethod
        def cast(cls, value):
            if not isinstance(value, dict):
                raise ValueError("Value {!r} is not a dict".format(value))
            return value

    class ResourceArch(sandboxsdk.parameters.SandboxStringParameter):
        name = 'resource_arch'
        description = 'Resource arch to be set on resource creation'
        required = True

    type = 'HTTP_UPLOAD'

    # TODO: SANDBOX-3450 Temporary deny execution on newly layouted hosts.
    client_tags = (
        ctc.Tag.GENERIC | ctc.Tag.STORAGE
    ) & ctc.Tag.Group.LINUX & ~ctc.Tag.LXC & ~ctc.Tag.NEW_LAYOUT

    input_parameters = [
        ResourceType,
        ResourceAttrs,
        ResourceArch,
    ]

    def handle_greetings(self, greets):
        ctx = self.ctx["upload"]
        ctx.pop("received", None)
        logging.info("Received %s", greets)
        assert greets.cnonce == ctx["cnonce"]
        ctx["proxy"] = {"node_id": greets.node_id, "request_id": greets.request_id}
        sandboxsdk.channel.channel.sandbox.set_task_context_value(self.id, "upload", ctx)

    def update_stream_status(self, received):
        self.ctx["upload"]["received"] = str(received)  # For XMLRPC stupid client
        sandboxsdk.channel.channel.sandbox.set_task_context_value(self.id, "upload", self.ctx["upload"])
        return received

    def on_prepare(self):
        ctx = self.ctx["upload"]
        assert ctm.Upload.VERSION >= ctx["version"]

        logging.debug("Creating a listening socket...")
        sh = SocketHandler()

        logging.info("Listening on %s:%r. Waiting for greetings.", *sh.addr)
        ctx["cnonce"] = random.getrandbits(24) & 0xFFFFF
        ctx["target"] = ":".join(map(str, sh.addr))
        sandboxsdk.channel.channel.sandbox.set_task_context_value(self.id, "upload", ctx)

        logging.debug("Registering a resource...")
        attrs = {"backup_task": True}
        attrs.update(string.parse_attrs(self.ctx[self.ResourceAttrs.name]))
        resource_id = self.ctx["resource_id"] = self.create_resource(
            self.descr,
            "UNKNOWN",
            self.ctx[self.ResourceType.name],
            self.ctx[self.ResourceArch.name],
            attrs
        ).id
        sandboxsdk.channel.channel.sandbox.set_task_context_value(self.id, "resource_id", str(resource_id))

        logging.debug("Waiting for proxy to connect...")
        self.handle_greetings(iter(sh).next())

    def on_execute(self):
        sh = SocketHandler()
        reader = TarballStreamReader(self, int(self.ctx["upload"]["amount"]), iter(sh))
        logging.info(
            "Consuming data stream of about %s and extract files to %r",
            common.utils.size2str(reader.expected), self.abs_path()
        )
        reader.extractall(self.abs_path())
        logging.debug(
            "Stream processing finished at %d bytes. Consuming the rest of the stream to provide correct checksum.",
            reader.received
        )
        while not reader.finished and reader.read(0xFFFFF):
            pass
        sha1 = self.ctx["upload"]["checksum"] = reader.sha1.hexdigest()
        self.ctx["upload"]["received"] = str(reader.received)  # Poor XMLRPC
        logging.info(
            "Received %s (%s bytes), SHA1: %s",
            common.utils.size2str(reader.received), reader.received, sha1
        )
        logging.info("Common name is %r", reader.common_prefix)

        if not reader.common_prefix:
            raise errors.SandboxTaskFailureError("No common prefix of uploaded files detected.")

        summary = iter(sh).next()
        assert isinstance(summary, sh.Proto.StreamSummary)
        if reader.received != summary.size:
            raise sandboxsdk.errors.SandboxTaskFailureError(
                "Received tarball stream of {} bytes, while it should be {} bytes.".format(
                    reader.received, summary.size
                )
            )
        if self.ctx["upload"]["version"] > 1 and summary.sha1 and sha1 != summary.sha1:
            raise sandboxsdk.errors.SandboxTaskFailureError(
                "Received tarball stream SHA1 '{}', while it should be '{}'.".format(
                    summary.sha1, sha1
                )
            )
        self.change_resource_basename(self.ctx["resource_id"], reader.common_prefix)
