# coding: utf-8

import six

from sandbox import sdk2
from sandbox.sdk2 import service_resources

from sandbox.common import errors
from sandbox.common.types import task as ctt


class MigrateSecrets(sdk2.Task):
    """
    Migrate Sandbox secrets to Yandex Vault (yav.yandex-team.ru).
    See https://wiki.yandex-team.ru/sandbox/yav/#migration for details.
    """

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 2048
        disk_space = 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # no shared caches

    class Parameters(sdk2.Parameters):
        description = "Migrate Sandbox secrets to Yav"
        kill_timeout = 180

        oauth_token = sdk2.parameters.Vault(
            "OAuth token for Yav",
            description=(
                "Either from Sandbox Vault (`<owner>:<name>`) or Yav (`sec-0abc@ver-0def[key]` or `sec-0abc[key]`)"
            ),
            required=True,
        )
        secrets = sdk2.parameters.Dict(
            "Keys and values for the new Yav secret",
            description="Values must be identifiers from Sandbox Vault (`<owner>:<name>`)",
            value_type=sdk2.parameters.String,
        )
        alias = sdk2.parameters.String(
            "Yav secret alias",
            description="Recognisable name for the new secret (can be changed in the future)",
            required=True,
        )

    def on_enqueue(self):
        if not self.Requirements.tasks_resource:
            self.Requirements.tasks_resource = service_resources.SandboxTasksBinary.find(
                attrs={"released": ctt.ReleaseStatus.STABLE, "task_type": self.type.name}
            ).first()

        if not self.Requirements.tasks_resource:
            raise errors.TaskError("Binary task resource is not found")

    def on_execute(self):
        # noinspection PyUnresolvedReferences
        from library.python.vault_client import errors as yav_errors
        # noinspection PyUnresolvedReferences
        from library.python.vault_client import instances

        if not self.Parameters.secrets:
            raise errors.TaskFailure("At least one secret is expected")

        futures = []
        with sdk2.task.Vault.batch:
            for key, secret_id in six.iteritems(self.Parameters.secrets):
                vault = sdk2.parameters.Vault.__decode__(secret_id)
                futures.append((key, secret_id, vault.data()))

        data = {}
        for key, secret_id, fut in futures:
            # noinspection PyBroadException
            try:
                data[key] = fut.result()
            except Exception:
                self.set_info("Could not get Sandbox secret: {}".format(secret_id))
                raise

        token = self.Parameters.oauth_token.data()

        yav_client = instances.Production(authorization="OAuth {}".format(token))

        try:
            secret_uuid = yav_client.create_secret(self.Parameters.alias)
            self.set_info(
                'New Yav secret: '
                '<a href="https://yav.yandex-team.ru/secret/{secret}">{secret}</a>'
                .format(secret=secret_uuid),
                do_escape=False
            )

            version_uuid = yav_client.create_secret_version(secret_uuid, data)
            self.set_info("New version: {} with {} key(s)".format(version_uuid, len(data)))

        except yav_errors.ClientError:
            self.set_info("Could not create Yav secret (is the OAuth-token valid?)")
            raise
