import os
import shutil
import logging
import datetime as dt

import pymongo
import pymongo.errors

from sandbox import common
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox import sdk2

from sandbox.projects.common import binary_task
from sandbox.projects.common import environments
import sandbox.projects.sandbox.resources as sb_resources


class ServiceMongoRestore(binary_task.LastBinaryTaskRelease, sdk2.ServiceTask):
    """ Restore the most recent backup of Sandbox production database. """

    """
    to avoid switching cluster into read-only mode and/or making it partially or completely unresponsive after restore,
    everything's done via temporary standalone database, which is run in a container
    """

    MONGO_HOST = "localhost"
    MONGOD_FOLDER = "mongod"

    class Requirements(sdk2.Requirements):
        disk_space = 2 * 400 * 1024  # 400Gb for backup and 400Gb for the database itself
        ram = 80 * 1024
        cores = 40
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LXC & ctc.Tag.LINUX_XENIAL
        environments = (environments.MongodbEnvironment(),)

    class Parameters(sdk2.Parameters):
        kill_timeout = 10 * 3600
        ext_params = binary_task.binary_release_parameters(stable=True)
        control_task_id = sdk2.parameters.Integer("Task id to check in restored DB")
        _container = sdk2.parameters.Container(
            "Container w/ MongoDB",
            resource_type=sb_resources.SandboxMongoContainer,
            required=True
        )

    @property
    def mongo_conn(self):
        return pymongo.MongoClient("mongodb://{}:22222".format(self.MONGO_HOST))

    @staticmethod
    def cleanup(processes, data_dir):
        with sdk2.helpers.ProgressMeter("cleaning things up"):
            logging.info("Performing cleanup...")
            for process in processes:
                process.kill()
            try:
                shutil.rmtree(os.path.join(os.getcwd(), str(data_dir)))
            except OSError:  # already removed
                logging.exception("Failed to remove directory %r", data_dir)

    def on_execute(self):
        from sandbox.scripts import mongo_tool
        from sandbox.scripts.mongo_tool import mongo_setup
        mongo_dir = sdk2.Path(os.getcwd()) / self.MONGOD_FOLDER
        mongo_dir.mkdir(exist_ok=True)
        logs_dir = self.log_path("mongod_logs")
        logs_dir.mkdir(exist_ok=True)
        today = dt.datetime.today().strftime("%Y%m%d")
        mongo_processes = []
        try:
            target_shards = mongo_tool.get_target_shards(mongo_setup.TESTING_SETUP)
            backup_dir = mongo_tool.download_latest_backups(today, shards=target_shards)
            self.set_info("Downloaded backup {}".format(os.path.basename(backup_dir)))
            os.rename(os.path.abspath(backup_dir), str(mongo_dir / "backup"))
            backup_dir = str(mongo_dir / "backup")

            self.set_info("Creating local mongo cluster (20 data shards, 1 config and 1 mongos)")
            mongo_processes = mongo_tool.create_local_cluster(str(mongo_dir), str(logs_dir))

            self.set_info("Cluster is ready, start restore")
            mongo_tool.restore_shards(
                setup=mongo_setup.TESTING_SETUP,
                date=today,
                directory=backup_dir,
                namespaces=["config.*", "sandbox.task"]
            )

            if not self.mongo_conn["sandbox"]["task"].count():
                raise common.errors.TaskFailure("Failed to restore {} collection".format("task"))
            last_task = self.mongo_conn["sandbox"]["task"].find().sort("_id", -1).limit(1)[0]["_id"]
            self.set_info("Mongo cluster restored with 'sandbox.task' collection (last id: {})".format(last_task))
            if self.Parameters.control_task_id:
                control_task_info = self.mongo_conn["sandbox"]["task"].find_one(self.Parameters.control_task_id)
                if not control_task_info:
                    raise common.errors.TaskFailure(
                        "Failed to restore control task {}".format(self.Parameters.control_task_id)
                    )
                self.set_info("Control task {} is restored".format(self.Parameters.control_task_id))
        finally:
            self.cleanup(mongo_processes, mongo_dir)

